/*
 * $Id: r128fb_accel.c,v 1.7 2003/08/19 07:11:05 obi Exp $
 *
 * ATI Rage 128 Framebuffer Driver
 *
 * Copyright (C) 2003 Andreas Oberritter <obi@saftware.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * This driver has been widely inspired by the Rage 128 drivers of
 * XFree86 4.x, Linux 2.4.x and svgalib 1.9.x. Thanks to everybody
 * who contributed to them.
 *
 */

#include <asm/errno.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/console.h>

#include "r128fb_accel.h"
#include "r128fb_inline.h"
#include "r128fb_reg.h"

#if 0
/*
 * D = destination bitmap
 * P = selected brush (pattern)
 * S = source bitmap
 *
 * a = bitwise AND
 * n = bitwise NOT
 * o = bitwise OR
 * x = bitwise exclusive OR
 */
static struct {
    int rop;
    int pattern;
} r128fb_rop[] = {
    { R128_ROP3_ZERO, R128_ROP3_ZERO }, /* GXclear        */
    { R128_ROP3_DSa,  R128_ROP3_DPa  }, /* Gxand          */
    { R128_ROP3_SDna, R128_ROP3_PDna }, /* GXandReverse   */
    { R128_ROP3_S,    R128_ROP3_P    }, /* GXcopy         */
    { R128_ROP3_DSna, R128_ROP3_DPna }, /* GXandInverted  */
    { R128_ROP3_D,    R128_ROP3_D    }, /* GXnoop         */
    { R128_ROP3_DSx,  R128_ROP3_DPx  }, /* GXxor          */
    { R128_ROP3_DSo,  R128_ROP3_DPo  }, /* GXor           */
    { R128_ROP3_DSon, R128_ROP3_DPon }, /* GXnor          */
    { R128_ROP3_DSxn, R128_ROP3_PDxn }, /* GXequiv        */
    { R128_ROP3_Dn,   R128_ROP3_Dn   }, /* GXinvert       */
    { R128_ROP3_SDno, R128_ROP3_PDno }, /* GXorReverse    */
    { R128_ROP3_Sn,   R128_ROP3_Pn   }, /* GXcopyInverted */
    { R128_ROP3_DSno, R128_ROP3_DPno }, /* GXorInverted   */
    { R128_ROP3_DSan, R128_ROP3_DPan }, /* GXnand         */
    { R128_ROP3_ONE,  R128_ROP3_ONE  }  /* GXset          */
};
#endif

/* Flush all dirty data in the Pixel Cache to memory. */
static inline
void r128fb_engine_flush(struct r128_info_rec *info)
{
	int i;

	OUTREGP(R128_PC_NGUI_CTLSTAT, R128_PC_FLUSH_ALL, ~R128_PC_FLUSH_ALL);

	for (i = 0; i < R128_TIMEOUT; i++)
		if (!(INREG(R128_PC_NGUI_CTLSTAT) & R128_PC_BUSY))
			break;
}

/* Reset graphics card to known state. */
static inline
void r128fb_engine_reset(struct r128_info_rec *info)
{
	u32 clock_cntl_index;
	u32 mclk_cntl;
	u32 gen_reset_cntl;

	r128fb_engine_flush(info);

	clock_cntl_index = INREG(R128_CLOCK_CNTL_INDEX);
	mclk_cntl = r128fb_inpll(info, R128_MCLK_CNTL);

	r128fb_outpll(info, R128_MCLK_CNTL, mclk_cntl | R128_FORCE_GCP | R128_FORCE_PIPE3D_CP);

	gen_reset_cntl = INREG(R128_GEN_RESET_CNTL);

	OUTREG(R128_GEN_RESET_CNTL, gen_reset_cntl | R128_SOFT_RESET_GUI);
	INREG(R128_GEN_RESET_CNTL);
	OUTREG(R128_GEN_RESET_CNTL, gen_reset_cntl & ~R128_SOFT_RESET_GUI);
	INREG(R128_GEN_RESET_CNTL);

	r128fb_outpll(info, R128_MCLK_CNTL, mclk_cntl);
	OUTREG(R128_CLOCK_CNTL_INDEX, clock_cntl_index);
	OUTREG(R128_GEN_RESET_CNTL, gen_reset_cntl);
}

/* The FIFO has 64 slots.  This routines waits until at least `entries' of
   these slots are empty. */
static inline
void r128fb_wait_for_fifo_function(struct r128_info_rec *info, u16 entries)
{
	int i;

	for (;;) {
		for (i = 0; i < R128_TIMEOUT; i++) {
			info->fifo_slots = INREG(R128_GUI_STAT) & R128_GUI_FIFOCNT_MASK;
			if (info->fifo_slots >= entries)
				return;
		}

		R128_DEBUG("FIFO timed out: %d entries, stat=0x%08x, probe=0x%08x\n",
			INREG(R128_GUI_STAT) & R128_GUI_FIFOCNT_MASK,
			INREG(R128_GUI_STAT),
			INREG(R128_GUI_PROBE));

		R128_ERR("FIFO timed out, resetting engine...\n");

		r128fb_engine_reset(info);
	}
}

void r128fb_wait_for_fifo(struct r128_info_rec *info, u16 entries)
{
	if (info->fifo_slots < entries)
		r128fb_wait_for_fifo_function(info, entries);
	info->fifo_slots -= entries;
}

/* Wait for the graphics engine to be completely idle: the FIFO has
   drained, the Pixel Cache is flushed, and the engine is idle.  This is a
   standard "sync" function that will make the hardware "quiescent". */
static inline
void r128fb_wait_for_idle(struct r128_info_rec *info)
{
	int i;

	r128fb_wait_for_fifo_function(info, 64);

	if ((info->irq_enabled) && (info->gen_int_cntl & R128_GUI_IDLE_INT)) {
		wait_event_interruptible(info->gui_idle_wait,
			!(INREG(R128_GUI_STAT) & R128_GUI_ACTIVE));
		r128fb_engine_flush(info);
	}
	else {
		for (;;) {
			for (i = 0; i < R128_TIMEOUT; i++) {
				if (!(INREG(R128_GUI_STAT) & R128_GUI_ACTIVE)) {
					r128fb_engine_flush(info);
					return;
				}
			}

			R128_DEBUG("Idle timed out: %d entries, stat=0x%08x, probe=0x%08x\n",
				INREG(R128_GUI_STAT) & R128_GUI_FIFOCNT_MASK,
				INREG(R128_GUI_STAT),
				INREG(R128_GUI_PROBE));

			R128_ERR("Idle timed out, resetting engine...\n");
			r128fb_engine_reset(info);
		}
	}
}

/* Initialize the acceleration hardware. */
static
void r128fb_engine_init(struct r128_info_rec *info)
{
	struct r128_layout *layout = &info->layout;

	R128_DEBUG("EngineInit (%d/%d)\n",
		layout->pixel_code, layout->bits_per_pixel);

	OUTREG(R128_SCALE_3D_CNTL, 0);
	r128fb_engine_reset(info);

	switch (layout->pixel_code) {
	case 8:  info->datatype = R128_GMC_DST_8BPP_CI; break;
	case 15: info->datatype = R128_GMC_DST_15BPP; break;
	case 16: info->datatype = R128_GMC_DST_16BPP; break;
	case 24: info->datatype = R128_GMC_DST_24BPP; break;
	case 32: info->datatype = R128_GMC_DST_32BPP; break;
	default:
		R128_ERR("Unknown depth/bpp = %d/%d (code = %d)\n",
			layout->depth, layout->bits_per_pixel,
			layout->pixel_code);
		break;
	}

	info->pitch = ((layout->display_width / 8) + 7) & ~7;
	
	if (layout->bits_per_pixel == 24)
		info->pitch *= 3;

	R128_DEBUG("Pitch for acceleration = %d\n", info->pitch);

	r128fb_wait_for_fifo(info, 2);

	/* setup engine offset registers */
	OUTREG(R128_DEFAULT_OFFSET, 0);

	/* setup engine pitch registers */
	OUTREG(R128_DEFAULT_PITCH, info->pitch);

	r128fb_wait_for_fifo(info, 4);
	OUTREG(R128_AUX_SC_CNTL, 0);
	OUTREG(R128_DEFAULT_SC_BOTTOM_RIGHT, (R128_DEFAULT_SC_RIGHT_MAX | R128_DEFAULT_SC_BOTTOM_MAX));
	OUTREG(R128_SC_TOP_LEFT, 0);
	OUTREG(R128_SC_BOTTOM_RIGHT, (R128_DEFAULT_SC_RIGHT_MAX | R128_DEFAULT_SC_BOTTOM_MAX));

	info->dp_gui_master_cntl = (info->datatype |
			R128_GMC_CLR_CMP_CNTL_DIS | R128_GMC_AUX_CLIP_DIS);

	r128fb_wait_for_fifo(info, 1);

	/* set the drawing controls registers */
	OUTREG(R128_DP_GUI_MASTER_CNTL, (info->dp_gui_master_cntl
			| R128_GMC_BRUSH_SOLID_COLOR
			| R128_GMC_SRC_DATATYPE_COLOR));

	r128fb_wait_for_fifo(info, 8);

	/* clear the line drawing registers */
	OUTREG(R128_DST_BRES_ERR, 0);
	OUTREG(R128_DST_BRES_INC, 0);
	OUTREG(R128_DST_BRES_DEC, 0);

	/* set brush color registers */
	OUTREG(R128_DP_BRUSH_FRGD_CLR, 0xffffffff);
	OUTREG(R128_DP_BRUSH_BKGD_CLR, 0x00000000);

	/* set source color registers */
	OUTREG(R128_DP_SRC_FRGD_CLR, 0xffffffff);
	OUTREG(R128_DP_SRC_BKGD_CLR, 0x00000000);

	/* default write mask */
	OUTREG(R128_DP_WRITE_MASK, 0xffffffff);

	r128fb_wait_for_fifo(info, 1);

#if defined(__BIG_ENDIAN)
	/* FIXME: this is a kludge for texture uploads in the 3D driver. Look at
	 * how the radeon driver handles HOST_DATA_SWAP if you want to implement
	 * CCE ImageWrite acceleration or anything needing this bit */
	OUTREGP(R128_DP_DATATYPE, R128_HOST_BIG_ENDIAN_EN, ~R128_HOST_BIG_ENDIAN_EN);
#else
	OUTREGP(R128_DP_DATATYPE, 0, ~R128_HOST_BIG_ENDIAN_EN);
#endif

	r128fb_wait_for_idle(info);

#if 0
	/* FIXME: enabling gui idle irq freezes machine */
	info->gen_int_cntl |= R128_GUI_IDLE_INT;
	OUTREG(R128_GEN_INT_STATUS, R128_GUI_IDLE_INT_AK);
	OUTREG(R128_GEN_INT_CNTL, info->gen_int_cntl);
#endif
}

static inline
void r128fb_cursor_hide(struct r128_info_rec *info)
{
	OUTREGP(R128_CRTC_GEN_CNTL, 0, ~R128_CRTC_CUR_EN);
}

static inline
void r128fb_cursor_show(struct r128_info_rec *info)
{
	OUTREGP(R128_CRTC_GEN_CNTL, R128_CRTC_CUR_EN, ~R128_CRTC_CUR_EN);
}

static inline
void r128fb_cursor_set_position(struct r128_info_rec *info, u16 x, u16 y)
{
	struct r128_cursor *cursor = &info->cursor;

	OUTREG(R128_CUR_HORZ_VERT_OFF, R128_CUR_LOCK);
	OUTREG(R128_CUR_HORZ_VERT_POSN, R128_CUR_LOCK | (x << 16) | y);
	OUTREG(R128_CUR_OFFSET, cursor->offset);
}

static inline
void r128fb_cursor_set_size_and_offset(struct r128_info_rec *info, u8 width, u8 height)
{
	struct r128_cursor *cursor = &info->cursor;
	struct r128_layout *layout = &info->layout;
	u32 offset;
	u32 save;
	u32 *d;
	u8 y;

	offset = ((layout->display_width * layout->display_height *
				layout->bits_per_pixel) + 255) & ~255;

	if ((cursor->width != width) || (cursor->height != height) ||
		(cursor->offset != offset)) {
		if ((width < 1) || (width > 64))
			return;
		if ((height < 1) || (height > 64))
			return;
		if (offset >= info->smem.len - 2048)
			return;

		d = (u32 *) &info->smem.ptr[offset];

		save = INREG(R128_CRTC_GEN_CNTL) & ~(3 << 20);
		save |= (2 << 20);
		OUTREG(R128_CRTC_GEN_CNTL, save & ~R128_CRTC_CUR_EN);

		for (y = 0; y < 64; y++) {
			*d++ = 0xffffffff;
			*d++ = 0xffffffff;
			if (y >= height) {
				*d++ = 0x00000000;
				*d++ = 0x00000000;
			}
			else if (width <= 32) {
				*d++ = (1ULL << width) - 1;
				*d++ = 0x00000000;
			}
			else {
				*d++ = 0xffffffff;
				*d++ = (1ULL << (width - 32)) - 1;
			}
		}

		for (y = 0; y < 64; y++) {
			*d++ = 0xffffffff;
			*d++ = 0xffffffff;
			*d++ = 0x00000000;
			*d++ = 0x00000000;
		}

		OUTREG(R128_CRTC_GEN_CNTL, save);

		cursor->width = width;
		cursor->height = height;
		cursor->offset = offset;
	}
}

static inline
void r128fb_rect_copy(struct r128_info_rec *info, int xa, int ya, int xb, int yb, int w, int h)
{
	u32 dp_cntl = 0;

	if (xa < xb) {
		xa += w - 1;
		xb += w - 1;
	}
	else {
		dp_cntl |= R128_DST_X_LEFT_TO_RIGHT;
	}

	if (ya < yb) {
		ya += h - 1;
		yb += h - 1;
	}
	else {
		dp_cntl |= R128_DST_Y_TOP_TO_BOTTOM;
	}

	r128fb_wait_for_fifo(info, 5);

	OUTREG(R128_DP_GUI_MASTER_CNTL, (info->dp_gui_master_cntl |
				R128_GMC_BRUSH_NONE |
				R128_GMC_SRC_DATATYPE_COLOR |
				R128_ROP3_S |
				R128_DP_SRC_SOURCE_MEMORY));
	OUTREG(R128_DP_CNTL, dp_cntl);
	OUTREG(R128_SRC_Y_X, (ya << 16) | xa);
	OUTREG(R128_DST_Y_X, (yb << 16) | xb);
	OUTREG(R128_DST_HEIGHT_WIDTH, (h << 16) | w);

	r128fb_wait_for_idle(info);
}

static inline
void r128fb_rect_rop(struct r128_info_rec *info, u32 rop, int x, int y, int w, int h, u32 color)
{
	r128fb_wait_for_fifo(info, 5);

	OUTREG(R128_DP_GUI_MASTER_CNTL, (info->dp_gui_master_cntl |
				R128_GMC_BRUSH_SOLID_COLOR |
				R128_GMC_SRC_DATATYPE_COLOR |
				rop));
	OUTREG(R128_DP_BRUSH_FRGD_CLR, color);
	OUTREG(R128_DP_CNTL, R128_DST_X_LEFT_TO_RIGHT | R128_DST_Y_TOP_TO_BOTTOM);
	OUTREG(R128_DST_Y_X, (y << 16) | x);
	OUTREG(R128_DST_WIDTH_HEIGHT, (w << 16) | h);

	r128fb_wait_for_idle(info);
}

static inline
void r128fb_rect_fill(struct r128_info_rec *info, int x, int y, int w, int h, u32 color)
{
	r128fb_rect_rop(info, R128_ROP3_P, x, y, w, h, color);
}

static inline
void r128fb_rect_invert(struct r128_info_rec *info, int x, int y, int w, int h, u32 color)
{
	r128fb_rect_rop(info, R128_ROP3_DPx, x, y, w, h, color);
}

/*
 * Linux Framebuffer API functions start here
 */

static
void r128fb_cfbX_bmove(struct display *disp, int ya, int xa, int yb, int xb, int h, int w)
{
	struct r128_info_rec *info = (struct r128_info_rec *) disp->fb_info;

	xa *= fontwidth(disp);
	ya *= fontheight(disp);
	xb *= fontwidth(disp);
	yb *= fontheight(disp);
	w *= fontwidth(disp);
	h *= fontheight(disp);

	r128fb_rect_copy(info, xa, ya, xb, yb, w, h);
}

static inline
void r128fb_cfbX_clear(struct display *disp, int x, int y, int w, int h, u32 color)
{
	r128fb_rect_fill((struct r128_info_rec *) disp->fb_info,
		x * fontwidth(disp),
		y * fontheight(disp),
		w * fontwidth(disp),
		h * fontheight(disp),
		color);
}

static
void r128fb_cfb8_clear(struct vc_data *vc, struct display *disp, int y, int x, int h, int w)
{
	u32 color = attr_bgcol_ec(disp, vc);

	color |= (color << 8);

	r128fb_cfbX_clear(disp, x, y, w, h, (color << 16) | color);
}

static
void r128fb_cfb16_clear(struct vc_data *vc, struct display *disp, int y, int x, int h, int w)
{
	u32 color = ((u16 *)disp->dispsw_data)[attr_bgcol_ec(disp, vc)];

	r128fb_cfbX_clear(disp, x, y, w, h, (color << 16) | color);
}

static
void r128fb_cfb32_clear(struct vc_data *vc, struct display *disp, int y, int x, int h, int w)
{
	u32 color = ((u32 *)disp->dispsw_data)[attr_bgcol_ec(disp, vc)];

	r128fb_cfbX_clear(disp, x, y, w, h, color);
}

static
void r128fb_cfb8_revc(struct display *disp, int x, int y)
{
	struct r128_info_rec *info = (struct r128_info_rec *) disp->fb_info;

	x *= fontwidth(disp);
	y *= fontheight(disp);

	r128fb_rect_invert(info, x, y, fontwidth(disp), fontheight(disp), 0x0f0f0f0f);
}

static	/* cfb16, cfb24, cfb32 */
void r128fb_cfbX_revc(struct display *disp, int x, int y)
{
	struct r128_info_rec *info = (struct r128_info_rec *) disp->fb_info;

	x *= fontwidth(disp);
	y *= fontheight(disp);

	r128fb_rect_invert(info, x, y, fontwidth(disp), fontheight(disp), 0xffffffff);
}

static
void r128fb_cfbX_cursor(struct display *disp, int mode, int x, int y)
{
	struct r128_info_rec *info = (struct r128_info_rec *) disp->fb_info;

	switch (mode) {
	case CM_DRAW:
		r128fb_cursor_show(info);
		break;
	case CM_ERASE:
		r128fb_cursor_hide(info);
		break;
	case CM_MOVE:
		r128fb_cursor_set_size_and_offset(info, fontwidth(disp), fontheight(disp));
		r128fb_cursor_set_position(info, x * fontwidth(disp), y * fontheight(disp));
		break;
	default:
		break;
	}
}

static
void r128fb_cfbX_clear_margins(struct vc_data *vc, struct display *disp, int bottom_only)
{
	struct r128_info_rec *info = (struct r128_info_rec *) disp->fb_info;
	unsigned int bottom_height, right_width;
	unsigned int bottom_start, right_start;
	unsigned int cell_h, cell_w;

	cell_w = fontwidth(disp);

	if (!cell_w)
		return;

	right_width = disp->var.xres % cell_w;
	right_start = disp->var.xres - right_width;

	if ((!bottom_only) && (right_width)) {
		r128fb_rect_fill(info, disp->var.xoffset + right_start, 0,
				right_width, disp->var.yres_virtual, 0);
	}

	cell_h = fontheight(disp);

	if (!cell_h)
		return;

	bottom_height = disp->var.yres % cell_h;

	if (bottom_height) {
		bottom_start = disp->var.yres - bottom_height;
		r128fb_rect_fill(info, disp->var.xoffset,
				disp->var.yoffset + bottom_start, right_start,
				bottom_height, 0);
	}
}

#if defined(FBCON_HAS_CFB8)
static struct display_switch r128fb_accel_cfb8 = {
	.setup = fbcon_cfb8_setup,
	.bmove = r128fb_cfbX_bmove,
	.clear = r128fb_cfb8_clear,
	.putc = fbcon_cfb8_putc,
	.putcs = fbcon_cfb8_putcs,
	.revc = r128fb_cfb8_revc,
	.cursor = r128fb_cfbX_cursor,
	.cursor = NULL,
	.set_font = NULL,
	.clear_margins = r128fb_cfbX_clear_margins,
	.fontwidthmask = FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(12) | FONTWIDTH(16)
};
#endif

#if defined(FBCON_HAS_CFB16)
static struct display_switch r128fb_accel_cfb16 = {
	.setup = fbcon_cfb16_setup,
	.bmove = r128fb_cfbX_bmove,
	.clear = r128fb_cfb16_clear,
	.putc = fbcon_cfb16_putc,
	.putcs = fbcon_cfb16_putcs,
	.revc = r128fb_cfbX_revc,
	//.cursor = r128fb_cfbX_cursor,
	.cursor = NULL,
	.set_font = NULL,
	.clear_margins = r128fb_cfbX_clear_margins,
	.fontwidthmask = FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(12) | FONTWIDTH(16)
};
#endif

#if defined(FBCON_HAS_CFB24)
static struct display_switch r128fb_accel_cfb24 = {
	.setup = fbcon_cfb24_setup,
	.bmove = r128fb_cfbX_bmove,
	.clear = r128fb_cfb32_clear,
	.putc = fbcon_cfb24_putc,
	.putcs = fbcon_cfb24_putcs,
	.revc = r128fb_cfbX_revc,
	//.cursor = r128fb_cfbX_cursor,
	.cursor = NULL,
	.set_font = NULL,
	.clear_margins = r128fb_cfbX_clear_margins,
	.fontwidthmask = FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(12) | FONTWIDTH(16)
};
#endif

#if defined(FBCON_HAS_CFB32)
static struct display_switch r128fb_accel_cfb32 = {
	.setup = fbcon_cfb32_setup,
	.bmove = r128fb_cfbX_bmove,
	.clear = r128fb_cfb32_clear,
	.putc = fbcon_cfb32_putc,
	.putcs = fbcon_cfb32_putcs,
	.revc = r128fb_cfbX_revc,
	//.cursor = r128fb_cfbX_cursor,
	.cursor = NULL,
	.set_font = NULL,
	.clear_margins = r128fb_cfbX_clear_margins,
	.fontwidthmask = FONTWIDTH(4) | FONTWIDTH(8) | FONTWIDTH(12) | FONTWIDTH(16)
};
#endif

static
void r128fb_accel_reset_modes(int index, u32 accel_flags)
{
	struct fb_var_screeninfo var;
	struct fb_info *fb = registered_fb[index];
	struct fb_ops *ops = fb->fbops;
	int i;

	for (i = 0; i < MAX_NR_CONSOLES; i++) {
		/* FIXME: won't work with multiple framebuffers */
		if (fb_display[i].conp) { /* && con2fb_map[i] == index) { */
			if (ops->fb_get_var(&var, i, fb)) {
				R128_ERR("fb_get_var failed\n");
				continue;
			}

			if (var.accel_flags == accel_flags)
				continue;

			var.accel_flags = accel_flags;
			var.activate = FB_ACTIVATE_NOW;

			if (ops->fb_set_var(&var, i, fb)) {
				R128_ERR("fb_set_var failed\n");
			}
		}
	}
}

int r128fb_accel_init_single(int index)
{
	struct r128_info_rec *info;

	if (!strncmp(registered_fb[index]->modename, "ATI Rage128", 11)) {
		info = (struct r128_info_rec *) registered_fb[index];
		info->dispsw_8bpp = &r128fb_accel_cfb8;
		info->dispsw_16bpp = &r128fb_accel_cfb16;
		info->dispsw_24bpp = &r128fb_accel_cfb24;
		info->dispsw_32bpp = &r128fb_accel_cfb32;
		info->engine_init = r128fb_engine_init;
		r128fb_accel_reset_modes(index, FB_ACCELF_TEXT);
		R128_INFO("enabled console acceleration on fb%d\n", index);
		return 1;
	}

	return 0;
}

void r128fb_accel_exit_single(int index)
{
	struct r128_info_rec *info;

	if (!strncmp(registered_fb[index]->modename, "ATI Rage128", 11)) {
		info = (struct r128_info_rec *) registered_fb[index];
		info->dispsw_8bpp = &fbcon_cfb8;
		info->dispsw_16bpp = &fbcon_cfb16;
		info->dispsw_24bpp = &fbcon_cfb24;
		info->dispsw_32bpp = &fbcon_cfb32;
		info->engine_init = NULL;
		r128fb_engine_reset(info);
		r128fb_accel_reset_modes(index, 0);
		R128_INFO("disabled console acceleration on fb%d\n", index);
	}
}

#if defined(R128FB_ACCEL_MODULE)
int r128fb_accel_init(void)
{
	int count = 0;
	int i;

	for (i = 0; i < num_registered_fb; i++)
		count += r128fb_accel_init_single(i);

	if (!count)
		return -ENODEV;

	return 0;
}

void r128fb_accel_exit(void)
{
	int i;

	for (i = 0; i < num_registered_fb; i++)
		r128fb_accel_exit_single(i);
}

module_init(r128fb_accel_init);
module_exit(r128fb_accel_exit);

MODULE_AUTHOR("Andreas Oberritter <obi@saftware.de>");
MODULE_DESCRIPTION("ATI Rage128 Framebuffer Acceleration Driver");
MODULE_LICENSE("GPL");
#endif /* R128FB_ACCEL_MODULE */
