/*
 * $Id: r128fb_base.c,v 1.18 2003/08/19 07:40:07 obi Exp $
 *
 * ATI Rage 128 Framebuffer Driver
 *
 * Copyright (C) 2003 Andreas Oberritter <obi@saftware.de>,
 *                    Andreas Hundt <andi@fischlustig.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * This driver has been widely inspired by the Rage 128 drivers of
 * XFree86 4.x, Linux 2.4.x and svgalib 1.9.x. Thanks to everybody
 * who contributed to them.
 *
 */

#include <linux/config.h>
#include <linux/console.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <asm/uaccess.h>

#if defined(CONFIG_MTRR)
#include <asm/mtrr.h>
#endif

#if defined(__powerpc__)
#include <asm/prom.h>
#include <asm/pci-bridge.h>
#endif

#include "r128fb_accel.h"
#include "r128fb_base.h"
#include "r128fb_inline.h"
#include "r128fb_reg.h"

static char *r128fb_name = "ATI Rage128";

static struct r128_ram_rec r128fb_ram_type[] = {
	{ 4, 4, 3, 3, 1, 3, 1, 16, 12 }, /* 128-bit SDR SGRAM 1:1	*/
	{ 4, 8, 3, 3, 1, 3, 1, 17, 13 }, /* 64-bit SDR SGRAM 1:1	*/
	{ 4, 4, 1, 2, 1, 2, 1, 16, 12 }, /* 64-bit SDR SGRAM 2:1	*/
	{ 4, 4, 3, 3, 2, 3, 1, 16, 12 }, /* 64-bit DDR SGRAM		*/
};

static struct fb_var_screeninfo r128fb_default = {
	.xres = 1024,
	.yres = 768,
	.xres_virtual = 1024,
	.yres_virtual = -1,
	.xoffset = 0,
	.yoffset = 0,
	.bits_per_pixel = 16,
	.grayscale = 0,
	.red = { .offset = 11, .length = 5, .msb_right = 0, },
	.green = { .offset = 5, .length = 6, .msb_right = 0, },
	.blue = { .offset = 0, .length = 5, .msb_right = 0, },
	.transp = { .offset = 0, .length = 0, .msb_right = 0, },
	.nonstd = 0,
	.activate = 0,
	.height = -1,
	.width = -1,
	.accel_flags = 0,
	.pixclock = 15385,
	.left_margin = 270,
	.right_margin = 33,
	.upper_margin = 30,
	.lower_margin = 2,
	.hsync_len = 17,
	.vsync_len = 6,
	.sync = FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,
	.vmode = FB_VMODE_NONINTERLACED,
};

static
void r128fb_irq(int irq, void *dev_id, struct pt_regs *fp)
{
	struct r128_info_rec *info = dev_id;
	u32 status = INREG(R128_GEN_INT_STATUS);

	if (status & R128_VSYNC_INT) {
		info->vsync_irq_count++;
		wake_up_interruptible(&info->vsync_wait);
	}

	if (status & R128_GUI_IDLE_INT) {
		wake_up_interruptible(&info->gui_idle_wait);
	}

	/* ack all interrupts */
	OUTREG(R128_GEN_INT_STATUS, status);
}

#if defined(__i386__)
static
int r128fb_map_rom_pci(struct r128_info_rec *info)
{
	struct pci_dev *dev = info->dev;
	struct resource *r;

	/* Fix from ATI for problem with Rage128 hardware not
	   leaving ROM enabled */
	OUTREGP(R128_MPP_TB_CONFIG, 0x04000000, 0x04ffffff);
	(void)INREG(R128_MPP_TB_CONFIG);

	/* no need to search for the ROM, just ask the card where it is. */
	r = &dev->resource[PCI_ROM_RESOURCE];

	/* assign the ROM an address if it doesn't have one */
	if (r->start == 0)
		pci_assign_resource(dev, PCI_ROM_RESOURCE);

	/* enable if needed */
	if (!(r->flags & PCI_ROM_ADDRESS_ENABLE)) {
		pci_write_config_dword(dev, dev->rom_base_reg, r->start | PCI_ROM_ADDRESS_ENABLE);
		r->flags |= PCI_ROM_ADDRESS_ENABLE;
	}

	info->bios.addr = r->start;
	info->bios.len = r->end - r->start + 1;
	info->bios.ptr = ioremap(info->bios.addr, info->bios.len);

	if (!info->bios.ptr)
		return -1;

	/* Very simple test to make sure it appeared */
	if ((readb(&info->bios.ptr[0]) != 0x55) ||
		(readb(&info->bios.ptr[1]) != 0xaa)) {
			iounmap(info->bios.ptr);
			return -1;
	}

	return 0;
}

static
int r128fb_map_rom_shadow(struct r128_info_rec *info)
{
	info->bios.addr = 0xc0000;
	info->bios.len = 0x10000;
	info->bios.ptr = ioremap(info->bios.addr, info->bios.len);

	if (!info->bios.ptr)
		return -1;

	/* Very simple test to make sure it appeared */
	if ((readb(&info->bios.ptr[0]) != 0x55) ||
		(readb(&info->bios.ptr[1]) != 0xaa)) {
			iounmap(info->bios.ptr);
			return -1;
	}

	return 0;
}

static
int r128fb_get_bios_parameters(struct r128_info_rec *info)
{
	u32 fp_header = 0;
	int i;

	if ((r128fb_map_rom_pci(info) < 0) &&
		(r128fb_map_rom_shadow(info) < 0))
			return -ENODEV;

	if (!info->has_panel_regs)
		return 0;

	for (i = 4; i < 0x10000 - 8; i++) {
		if ((R128_BIOS8(i) == 'M') &&
			(R128_BIOS8(i + 1) == '3') &&
			(R128_BIOS8(i + 2) == ' ') &&
			(R128_BIOS8(i + 3) == ' ') &&
			(R128_BIOS8(i + 4) == ' ') &&
			(R128_BIOS8(i + 5) == ' ') &&
			(R128_BIOS8(i + 6) == ' ') &&
			(R128_BIOS8(i + 7) == ' ')) {
				fp_header = i - 2;
				break;
		}
	}

	if (!fp_header)
		return 0;

	for (i = fp_header + 20; i < fp_header + 84; i += 2) {
		if (R128_BIOS16(i) != 0) {
			info->fp_bios_start = R128_BIOS16(i);
			break;
		}
	}

	if (!info->fp_bios_start)
		return 0;

	info->panel_xres = R128_BIOS16(info->fp_bios_start + 25);
	info->panel_yres = R128_BIOS16(info->fp_bios_start + 27);
	info->panel_pwr_dly = R128_BIOS8(info->fp_bios_start + 56);

	if ((!info->panel_xres) || (!info->panel_yres)) {
		info->has_panel_regs = 0;
		return 0;
	}

	R128_NOTICE("panel_id = ");
	for (i = 1; i <= 24; i++)
		printk("%c", R128_BIOS8(info->fp_bios_start + i));
	printk("\n");

	R128_NOTICE("panel_type = ");
		
	i = R128_BIOS16(info->fp_bios_start + 29);

	if (i & 1)
		printk("Color, ");
	else
		printk("Monochrome, ");
		
	if (i & 2)
		printk("Dual(split), ");
	else
		printk("Single, ");
		
	switch ((i >> 2) & 0x3f) {
	case 0:
		printk("STN");
		break;
	case 1:
		printk("TFT");
		break;
	case 2:
		printk("Active STN");
		break;
	case 3:
		printk("EL");
		break;
	case 4:
		printk("Plasma");
		break;
	default:
		printk("unknown");
		break;
	}

	printk("\n");

	R128_NOTICE("panel_interface = ");

	if (R128_BIOS8(info->fp_bios_start + 61) & 1)
		printk("LVDS");
	else
		printk("unknown");

	printk("\n");

	return 0;
}
#elif defined(__powerpc__)
static
int r128fb_get_bios_parameters(struct r128_info_rec *info)
{
        struct device_node *dp;
	unsigned int *pvalue = NULL;
	struct property *pro = NULL;

	if (!info->has_panel_regs)
		return 0;

	/* these are defaults, if reading from OF fails */
	info->panel_xres = 1024;
	info->panel_yres = 768;
	info->panel_pwr_dly = 10;

	dp = pci_device_to_OF_node(info->dev);
	if (!dp)
		return 0;

	/*
	 * if the device has a child, it seems to contain the important
	 * properties ... at least on my machine ...
	 */
	if (dp->child)
		dp = dp->child;
#if 0
	/* DEBUG: dump all properties */
	pro = dp->properties;
	
	while(pro) {
	     printk("%s \n",pro->name);
	     pro = pro->next;
	}
#endif
	pvalue = (unsigned int *)get_property(dp, "width", NULL);

	if (pvalue) {
		info->panel_xres = *pvalue;
		R128_NOTICE("width = %d (value from OpenFirmware)\n",
			info->panel_xres);
	}
	else {
		R128_NOTICE("could not read width from OpenFirmware, default to %d\n", info->panel_xres);
	}

	pvalue = (unsigned int *)get_property(dp, "height", NULL);

	if (pvalue) {
		info->panel_yres = *pvalue;
		R128_NOTICE("height = %d (value from OpenFirmware)\n",
			info->panel_yres);
	}
	else {
		R128_NOTICE("could not read height from OpenFirmware, default to %d\n", info->panel_yres);
	}

	return 0;
}
#else
static
int r128fb_get_bios_parameters(struct r128_info_rec *info)
{
	return 0;
}
#endif

#if defined(__i386__)
static
int r128fb_get_pll_parameters(struct r128_info_rec *info)
{
	struct r128_pll_rec *pll = &info->pll;
	u16 bios_header = R128_BIOS16(0x48);
	u16 pll_info_block = R128_BIOS16(bios_header + 0x30);

	R128_INFO("Header at 0x%04x; PLL Information at 0x%04x\n",
			bios_header, pll_info_block);

	pll->reference_freq = R128_BIOS16(pll_info_block + 0x0e);
	pll->reference_div = R128_BIOS16(pll_info_block + 0x10);
	pll->min_pll_freq = R128_BIOS32(pll_info_block + 0x12);
	pll->max_pll_freq = R128_BIOS32(pll_info_block + 0x16);
	pll->xclk = R128_BIOS16(pll_info_block + 0x08);

	R128_INFO("PLL parameters: rf=%d rd=%d min=%d max=%d; xclk=%d\n",
			pll->reference_freq,
			pll->reference_div,
			pll->min_pll_freq,
			pll->max_pll_freq,
			pll->xclk);

	return 0;
}
#else
static
int r128fb_get_pll_parameters(struct r128_info_rec *info)
{
	/* 
	 * this function assumes Open Firmware has set up pll correctly 
	 */
	u32 x_mpll_ref_fb_div;
	u32 xclk_cntl;
	u32 Nx, M;
	unsigned PostDivSet[] = { 0, 1, 2, 4, 8, 3, 6, 12 };

	struct r128_pll_rec *pll = &info->pll;

	pll->reference_freq = 2950;
	pll->reference_div = r128fb_inpll(info, R128_PPLL_REF_DIV) 
	                     & R128_PPLL_REF_DIV_MASK;
	pll->min_pll_freq = 12500;
	pll->max_pll_freq = 25000;

	x_mpll_ref_fb_div = r128fb_inpll(info, R128_X_MPLL_REF_FB_DIV);
	xclk_cntl = r128fb_inpll(info, R128_XCLK_CNTL) & 0x7; 
	Nx = (x_mpll_ref_fb_div & 0x00ff00) >> 8;
	M  = x_mpll_ref_fb_div & 0x0000ff;

	pll->xclk = r128fb_div(2 * Nx * pll->reference_freq,
	                       M * PostDivSet[xclk_cntl]);

	R128_INFO("PLL parameters: rf=%d rd=%d min=%d max=%d; xclk=%d\n",
			pll->reference_freq,
			pll->reference_div,
			pll->min_pll_freq,
			pll->max_pll_freq,
			pll->xclk);

	return 0;
}
#endif

static
int r128fb_wait_for_vertical_sync(struct r128_info_rec *info)
{
	int i;

	if ((info->irq_enabled) && (info->gen_int_cntl & R128_VSYNC_INT)) {
		info->vsync_irq_count = 0;
		return wait_event_interruptible(info->vsync_wait, info->vsync_irq_count != 0);
	}
	else {
		OUTREG(R128_GEN_INT_STATUS, R128_VSYNC_INT_AK);
		for (i = 0; i < R128_TIMEOUT; i++)
			if (INREG(R128_GEN_INT_STATUS) & R128_VSYNC_INT)
				break;
	}

	return 0;
}

static
void r128fb_set_backlight(struct r128_info_rec *info, int enable)
{
	u32 lvds_gen_cntl = INREG(R128_LVDS_GEN_CNTL);
	u32 lvds_state_mask = (R128_LVDS_ON | R128_LVDS_DISPLAY_DIS |
			R128_LVDS_EN | R128_LVDS_DIGON | R128_LVDS_BLON);

	lvds_gen_cntl |= R128_LVDS_BLON;

	if (enable) {
		lvds_gen_cntl |= R128_LVDS_DIGON;

		if (!(lvds_gen_cntl & R128_LVDS_ON)) {
			lvds_gen_cntl &= ~R128_LVDS_BLON;
			OUTREG(R128_LVDS_GEN_CNTL, lvds_gen_cntl);
			(void)INREG(R128_LVDS_GEN_CNTL);
			mdelay(10);
			lvds_gen_cntl |= R128_LVDS_BLON;
			OUTREG(R128_LVDS_GEN_CNTL, lvds_gen_cntl);
		}

		lvds_gen_cntl |= (R128_LVDS_ON | R128_LVDS_EN);
		lvds_gen_cntl &= ~R128_LVDS_DISPLAY_DIS;
	}
	else {
		lvds_gen_cntl |= R128_LVDS_DISPLAY_DIS;
		OUTREG(R128_LVDS_GEN_CNTL, lvds_gen_cntl);
		udelay(10);
		lvds_gen_cntl &= ~(R128_LVDS_ON | R128_LVDS_EN |
				R128_LVDS_BLON | R128_LVDS_DIGON);
	}

	OUTREG(R128_LVDS_GEN_CNTL, lvds_gen_cntl);
	info->saved_reg.lvds_gen_cntl &= ~lvds_state_mask;
	info->saved_reg.lvds_gen_cntl |= (lvds_gen_cntl & lvds_state_mask);
}

static
void r128fb_set_dpms_crtc(struct r128_info_rec *info, int mode)
{
	u32 mask = R128_CRTC_DISPLAY_DIS | R128_CRTC_HSYNC_DIS |
			R128_CRTC_VSYNC_DIS;

	switch (mode) {
	case -1:
	case VESA_NO_BLANKING:
		/* Screen: On; HSync: On, VSync: On */
		OUTREGP(R128_CRTC_EXT_CNTL, 0, ~mask);
		break;
	case VESA_VSYNC_SUSPEND:
		/* Screen: Off; HSync: Off, VSync: On */
		OUTREGP(R128_CRTC_EXT_CNTL, R128_CRTC_DISPLAY_DIS |
			R128_CRTC_HSYNC_DIS, ~mask);
		break;
	case VESA_HSYNC_SUSPEND:
		/* Screen: Off; HSync: On, VSync: Off */
		OUTREGP(R128_CRTC_EXT_CNTL, R128_CRTC_DISPLAY_DIS |
			R128_CRTC_VSYNC_DIS, ~mask);
		break;
	case VESA_POWERDOWN:
		/* Screen: Off; HSync: Off, VSync: Off */
		OUTREGP(R128_CRTC_EXT_CNTL, mask, ~mask);
		break;
	default:
		R128_ERR("unknown dpms mode %d\n", mode);
		break;
	}
}

static
void r128fb_set_dpms_fp(struct r128_info_rec *info, int mode)
{
	u32 mask = R128_LVDS_DISPLAY_DIS;

	switch (mode) {
	case -1:
	case VESA_NO_BLANKING:
		/* Screen: On; HSync: On, VSync: On */
		OUTREGP(R128_LVDS_GEN_CNTL, 0, ~mask);
		r128fb_set_backlight(info, 1);
		break;
	case VESA_VSYNC_SUSPEND:
	case VESA_HSYNC_SUSPEND:
		break;
	case VESA_POWERDOWN:
		/* Screen: Off; HSync: Off, VSync: Off */
		OUTREGP(R128_LVDS_GEN_CNTL, mask, ~mask);
		r128fb_set_backlight(info, 0);
		break;
	default:
		R128_ERR("unknown dpms mode %d\n", mode);
		break;
	}
}

static inline
void r128fb_blank(struct r128_info_rec *info, int blank)
{
	if ((!info->has_panel_regs) ||
		(info->bios_display == R128_BIOS_DISPLAY_CRT))
			r128fb_set_dpms_crtc(info, blank);
	else if ((info->has_panel_regs) || 
		(info->bios_display == R128_BIOS_DISPLAY_FP))
			r128fb_set_dpms_fp(info, blank);
}

static
void r128fb_restore_common_registers(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	OUTREG(R128_FP_GEN_CNTL, restore->fp_gen_cntl | R128_FP_BLANK_DIS);

	OUTREG(R128_OVR_CLR, restore->ovr_clr);
	OUTREG(R128_OVR_WID_LEFT_RIGHT, restore->ovr_wid_left_right);
	OUTREG(R128_OVR_WID_TOP_BOTTOM, restore->ovr_wid_top_bottom);
	OUTREG(R128_OV0_SCALE_CNTL, restore->ov0_scale_cntl);
	OUTREG(R128_MPP_TB_CONFIG, restore->mpp_tb_config);
	OUTREG(R128_MPP_GP_CONFIG, restore->mpp_gp_config);
	OUTREG(R128_SUBPIC_CNTL, restore->subpic_cntl);
	OUTREG(R128_VIPH_CONTROL, restore->viph_control);
	OUTREG(R128_I2C_CNTL_1, restore->i2c_cntl_1);
	OUTREG(R128_GEN_INT_CNTL, restore->gen_int_cntl);
	OUTREG(R128_CAP0_TRIG_CNTL, restore->cap0_trig_cntl);
	OUTREG(R128_CAP1_TRIG_CNTL, restore->cap1_trig_cntl);

	OUTREG(R128_BUS_CNTL, restore->bus_cntl);
	OUTREG(R128_CONFIG_CNTL, restore->config_cntl);

	if (!restore->gen_int_cntl)
		info->irq_enabled = 0;
}

static
void r128fb_restore_crtc_registers(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	OUTREG(R128_CRTC_GEN_CNTL, restore->crtc_gen_cntl);

	OUTREGP(R128_CRTC_EXT_CNTL, restore->crtc_ext_cntl,
		R128_CRTC_VSYNC_DIS |
		R128_CRTC_HSYNC_DIS |
		R128_CRTC_DISPLAY_DIS);

	OUTREGP(R128_DAC_CNTL, restore->dac_cntl,
		R128_DAC_RANGE_CNTL | R128_DAC_BLANKING);

	OUTREG(R128_CRTC_H_TOTAL_DISP, restore->crtc_h_total_disp);
	OUTREG(R128_CRTC_H_SYNC_STRT_WID, restore->crtc_h_sync_strt_wid);
	OUTREG(R128_CRTC_V_TOTAL_DISP, restore->crtc_v_total_disp);
	OUTREG(R128_CRTC_V_SYNC_STRT_WID, restore->crtc_v_sync_strt_wid);
	OUTREG(R128_CRTC_OFFSET, restore->crtc_offset);
	OUTREG(R128_CRTC_OFFSET_CNTL, restore->crtc_offset_cntl);
	OUTREG(R128_CRTC_PITCH, restore->crtc_pitch);
}

static
void r128fb_restore_fp_registers(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	u32 tmp;

	OUTREG(R128_FP_HORZ_STRETCH, restore->fp_horz_stretch);
	OUTREG(R128_FP_VERT_STRETCH, restore->fp_vert_stretch);
	OUTREG(R128_FP_CRTC_H_TOTAL_DISP, restore->fp_crtc_h_total_disp);
	OUTREG(R128_FP_CRTC_V_TOTAL_DISP, restore->fp_crtc_v_total_disp);
	OUTREG(R128_FP_H_SYNC_STRT_WID, restore->fp_h_sync_strt_wid);
	OUTREG(R128_FP_V_SYNC_STRT_WID, restore->fp_v_sync_strt_wid);
	OUTREG(R128_TMDS_CRC, restore->tmds_crc);
	OUTREG(R128_FP_PANEL_CNTL, restore->fp_panel_cntl);
	OUTREG(R128_FP_GEN_CNTL, restore->fp_gen_cntl & ~R128_FP_BLANK_DIS);

	if (info->is_dfp)
		return;

	tmp = INREG(R128_LVDS_GEN_CNTL);

	if ((tmp & (R128_LVDS_ON | R128_LVDS_BLON)) == (restore->lvds_gen_cntl & (R128_LVDS_ON | R128_LVDS_BLON))) {
		OUTREG(R128_LVDS_GEN_CNTL, restore->lvds_gen_cntl);
	}
	else if (restore->lvds_gen_cntl & (R128_LVDS_ON | R128_LVDS_BLON)) {
		OUTREG(R128_LVDS_GEN_CNTL, restore->lvds_gen_cntl & ~R128_LVDS_BLON);
		mdelay(info->panel_pwr_dly);
		OUTREG(R128_LVDS_GEN_CNTL, restore->lvds_gen_cntl);
	}
	else {
		OUTREG(R128_LVDS_GEN_CNTL, restore->lvds_gen_cntl | R128_LVDS_BLON);
		mdelay(info->panel_pwr_dly);
		OUTREG(R128_LVDS_GEN_CNTL, restore->lvds_gen_cntl);
	}
}

static
void r128fb_restore_pll_registers(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	/*
	 * from xfree86/radeonfb:
	 * A temporal workaround for the occational blanking on certain laptop
	 * panels. This appears to related to the PLL divider registers (fail
	 * to lock?). It occurs even when all dividers are the same with their
	 * old settings. In this case we really don't need to fiddle with PLL
	 * registers. By doing this we can avoid the blanking problem with some
	 * panels.
	 */
	if (1) {
		u32 ppll_ref_div = r128fb_inpll(info, R128_PPLL_REF_DIV) &
			R128_PPLL_REF_DIV_MASK;
		u32 ppll_div_3 = r128fb_inpll(info, R128_PPLL_DIV_3) &
			(R128_PPLL_POST3_DIV_MASK | R128_PPLL_FB3_DIV_MASK);
		if ((restore->ppll_ref_div == ppll_ref_div) &&
			(restore->ppll_div_3 == ppll_div_3))
				return;
	}

	OUTREGP(R128_CLOCK_CNTL_INDEX, R128_PLL_DIV_SEL, 0xffff);

	r128fb_outpllp(info,
		R128_PPLL_CNTL,
		R128_PPLL_RESET |
		R128_PPLL_ATOMIC_UPDATE_EN |
		R128_PPLL_VGA_ATOMIC_UPDATE_EN,
		0xffff);

	r128fb_pll_wait_read_update_complete(info);
	r128fb_outpllp(info, R128_PPLL_REF_DIV, restore->ppll_ref_div, ~R128_PPLL_REF_DIV_MASK);
	r128fb_pll_write_update(info);

	r128fb_pll_wait_read_update_complete(info);
	r128fb_outpllp(info, R128_PPLL_DIV_3, restore->ppll_div_3, ~R128_PPLL_FB3_DIV_MASK);
	r128fb_pll_write_update(info);
	r128fb_outpllp(info, R128_PPLL_DIV_3, restore->ppll_div_3, ~R128_PPLL_POST3_DIV_MASK);
	r128fb_pll_write_update(info);

	r128fb_outpll(info, R128_HTOTAL_CNTL, restore->htotal_cntl);

	r128fb_outpllp(info, R128_PPLL_CNTL, 0, ~R128_PPLL_RESET);

	R128_DEBUG("Wrote: 0x%08x 0x%08x 0x%08x (0x%08x)\n",
			restore->ppll_ref_div,
			restore->ppll_div_3,
			restore->htotal_cntl,
			r128fb_inpll(info, R128_PPLL_CNTL));

	R128_DEBUG("Wrote: rd=%d, fd=%d, pd=%d\n",
			restore->ppll_ref_div & R128_PPLL_REF_DIV_MASK,
			restore->ppll_div_3 & R128_PPLL_FB3_DIV_MASK,
			(restore->ppll_div_3 & R128_PPLL_POST3_DIV_MASK) >> 16);
}

static
void r128fb_restore_dda_registers(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	OUTREG(R128_DDA_CONFIG, restore->dda_config);
	OUTREG(R128_DDA_ON_OFF, restore->dda_on_off);
}

static
void r128fb_restore_palette(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	int i;

	if (!restore->palette_valid)
		return;

	if ((info->has_panel_regs) || (info->is_dfp))
		r128fb_pal_select(info, 0);

	r128fb_outpal_start(info, 0);

	for (i = 0; i < 256; i++)
		r128fb_outpal_next32(info, restore->palette[i]);
}

static
void r128fb_restore_mode(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	r128fb_restore_common_registers(info, restore);
	r128fb_restore_crtc_registers(info, restore);
	if ((!(info->has_panel_regs)) || (info->bios_display == R128_BIOS_DISPLAY_CRT))
		r128fb_restore_pll_registers(info, restore);
	r128fb_restore_dda_registers(info, restore);
	if ((info->has_panel_regs) || (info->is_dfp))
		r128fb_restore_fp_registers(info, restore);
	r128fb_restore_palette(info, restore);
}

static
void r128fb_restore(struct r128_info_rec *info, struct r128_save_rec *restore)
{
	r128fb_blank(info, VESA_VSYNC_SUSPEND);

	OUTREG(R128_AMCGPIO_MASK, restore->amcgpio_mask);
	OUTREG(R128_AMCGPIO_EN_REG, restore->amcgpio_en_reg);
	OUTREG(R128_CLOCK_CNTL_INDEX, restore->clock_cntl_index);
	OUTREG(R128_GEN_RESET_CNTL, restore->gen_reset_cntl);
	OUTREG(R128_DP_DATATYPE, restore->dp_datatype);

	r128fb_restore_mode(info, restore);

	if (info->gen_int_cntl) {
		/* re-enable interrupts before waiting for vsync */
		OUTREG(R128_GEN_INT_CNTL, info->gen_int_cntl);
		info->irq_enabled = 1;
	}

	r128fb_wait_for_vertical_sync(info);
	r128fb_blank(info, VESA_NO_BLANKING);

	/* re-enable i2c */
	OUTREG(R128_I2C_CNTL_1, info->i2c_cntl_1);
}

static
void r128fb_save_common_registers(struct r128_info_rec *info, struct r128_save_rec *save)
{
	save->ovr_clr = INREG(R128_OVR_CLR);
	save->ovr_wid_left_right = INREG(R128_OVR_WID_LEFT_RIGHT);
	save->ovr_wid_top_bottom = INREG(R128_OVR_WID_TOP_BOTTOM);
	save->ov0_scale_cntl = INREG(R128_OV0_SCALE_CNTL);
	save->mpp_tb_config = INREG(R128_MPP_TB_CONFIG);
	save->mpp_gp_config = INREG(R128_MPP_GP_CONFIG);
	save->subpic_cntl = INREG(R128_SUBPIC_CNTL);
	save->viph_control = INREG(R128_VIPH_CONTROL);
	save->i2c_cntl_1 = INREG(R128_I2C_CNTL_1);
	save->gen_int_cntl = INREG(R128_GEN_INT_CNTL);
	save->cap0_trig_cntl = INREG(R128_CAP0_TRIG_CNTL);
	save->cap1_trig_cntl = INREG(R128_CAP1_TRIG_CNTL);
	save->bus_cntl = INREG(R128_BUS_CNTL);
	save->config_cntl = INREG(R128_CONFIG_CNTL);
}

static
void r128fb_save_crtc_registers(struct r128_info_rec *info, struct r128_save_rec *save)
{
	save->crtc_gen_cntl = INREG(R128_CRTC_GEN_CNTL);
	save->crtc_ext_cntl = INREG(R128_CRTC_EXT_CNTL);
	save->dac_cntl = INREG(R128_DAC_CNTL);
	save->crtc_h_total_disp = INREG(R128_CRTC_H_TOTAL_DISP);
	save->crtc_h_sync_strt_wid = INREG(R128_CRTC_H_SYNC_STRT_WID);
	save->crtc_v_total_disp = INREG(R128_CRTC_V_TOTAL_DISP);
	save->crtc_v_sync_strt_wid = INREG(R128_CRTC_V_SYNC_STRT_WID);
	save->crtc_offset = INREG(R128_CRTC_OFFSET);
	save->crtc_offset_cntl = INREG(R128_CRTC_OFFSET_CNTL);
	save->crtc_pitch = INREG(R128_CRTC_PITCH);
}

static
void r128fb_save_fp_registers(struct r128_info_rec *info, struct r128_save_rec *save)
{
	save->crtc2_gen_cntl = INREG(R128_CRTC2_GEN_CNTL);
	save->fp_crtc_h_total_disp = INREG(R128_FP_CRTC_H_TOTAL_DISP);
	save->fp_crtc_v_total_disp = INREG(R128_FP_CRTC_V_TOTAL_DISP);
	save->fp_gen_cntl = INREG(R128_FP_GEN_CNTL);
	save->fp_h_sync_strt_wid = INREG(R128_FP_H_SYNC_STRT_WID);
	save->fp_horz_stretch = INREG(R128_FP_HORZ_STRETCH);
	save->fp_panel_cntl = INREG(R128_FP_PANEL_CNTL);
	save->fp_v_sync_strt_wid = INREG(R128_FP_V_SYNC_STRT_WID);
	save->fp_vert_stretch = INREG(R128_FP_VERT_STRETCH);
	save->lvds_gen_cntl = INREG(R128_LVDS_GEN_CNTL);
	save->tmds_crc = INREG(R128_TMDS_CRC);
	save->tmds_transmitter_cntl = INREG(R128_TMDS_TRANSMITTER_CNTL);
}

static
void r128fb_save_pll_registers(struct r128_info_rec *info, struct r128_save_rec *save)
{
	save->ppll_ref_div = r128fb_inpll(info, R128_PPLL_REF_DIV);
	save->ppll_div_3 = r128fb_inpll(info, R128_PPLL_DIV_3);
	save->htotal_cntl = r128fb_inpll(info, R128_HTOTAL_CNTL);

	R128_DEBUG("Read: 0x%08x 0x%08x 0x%08x\n",
			save->ppll_ref_div,
			save->ppll_div_3,
			save->htotal_cntl);

	R128_DEBUG("Read: rd=%d, fd=%d, pd=%d\n",
			save->ppll_ref_div & R128_PPLL_REF_DIV_MASK,
			save->ppll_div_3 & R128_PPLL_FB3_DIV_MASK,
			(save->ppll_div_3 & R128_PPLL_POST3_DIV_MASK) >> 16);
}

static
void r128fb_save_dda_registers(struct r128_info_rec *info, struct r128_save_rec *save)
{
	save->dda_config = INREG(R128_DDA_CONFIG);
	save->dda_on_off = INREG(R128_DDA_ON_OFF);
}

static
void r128fb_save_palette(struct r128_info_rec *info, struct r128_save_rec *save)
{
	int i;

	if ((info->has_panel_regs) || (info->is_dfp))
		r128fb_pal_select(info, 0);

	r128fb_inpal_start(info, 0);

	for (i = 0; i < 256; i++)
		save->palette[i] = r128fb_inpal_next(info);

	save->palette_valid = 1;
}

static
void r128fb_save_mode(struct r128_info_rec *info, struct r128_save_rec *save)
{
	r128fb_save_common_registers(info, save);
	r128fb_save_crtc_registers(info, save);
	if ((info->has_panel_regs) || (info->is_dfp))
		r128fb_save_fp_registers(info, save);
	r128fb_save_pll_registers(info, save);
	r128fb_save_dda_registers(info, save);
	r128fb_save_palette(info, save);
}

static
void r128fb_save(struct r128_info_rec *info, struct r128_save_rec *save)
{
	r128fb_save_mode(info, save);

	save->dp_datatype = INREG(R128_DP_DATATYPE);
	save->gen_reset_cntl = INREG(R128_GEN_RESET_CNTL);
	save->clock_cntl_index = INREG(R128_CLOCK_CNTL_INDEX);
	save->amcgpio_en_reg = INREG(R128_AMCGPIO_EN_REG);
	save->amcgpio_mask = INREG(R128_AMCGPIO_MASK);
}

static
void r128fb_init_common_registers(struct r128_info_rec *info, struct r128_save_rec *save)
{
	save->ovr_clr = 0;
	save->ovr_wid_left_right = 0;
	save->ovr_wid_top_bottom = 0;
	save->ov0_scale_cntl = 0;
	save->mpp_tb_config = 0;
	save->mpp_gp_config = 0;
	save->subpic_cntl = 0;
	save->viph_control = 0;
	save->i2c_cntl_1 = 0;
	save->gen_int_cntl = 0;
	save->cap0_trig_cntl = 0;
	save->cap1_trig_cntl = 0;
	save->bus_cntl = info->bus_cntl;

	if (save->bus_cntl & (R128_BUS_WRT_BURST | R128_BUS_READ_BURST))
		save->bus_cntl |= R128_BUS_RD_DISCARD_EN | R128_BUS_RD_ABORT_EN;
}

static
int r128fb_init_crtc_registers(struct r128_info_rec *info, struct r128_save_rec *save, const struct fb_var_screeninfo *var)
{
	struct r128_layout *layout = &info->layout;
	u32 xres = var->xres;
	u32 yres = var->yres;
	u32 vxres = var->xres_virtual;
	u32 vyres = var->yres_virtual;
	u32 xoffset = var->xoffset;
	u32 yoffset = var->yoffset;
	u32 bpp = var->bits_per_pixel;
	u16 h_total;
	u16 v_total;
	u8 format;
	u8 depth;
	u8 hsync_fudge_default[] = { 0x00, 0x12, 0x09, 0x09, 0x06, 0x05 };
	u8 hsync_fudge_fp[] = { 0x12, 0x11, 0x09, 0x09, 0x05, 0x05 };
	u8 hsync_fudge_fp_crt[] = { 0x12, 0x10, 0x08, 0x08, 0x04, 0x04 };
	u8 hsync_fudge;
	u8 hsync_wid;
	u16 hsync_start;
	u8 vsync_wid;
	u16 vsync_start;

	if (bpp != 16)
		depth = bpp;
	else
		depth = (var->green.length == 6) ? 16 : 15;

	/* convert (and round up) and validate */
	xres = (xres + 7) & ~7;
	xoffset = (xoffset + 7) & ~7;

	if ((vxres == -1) && (vyres == -1)) {
		/* TODO: steal code from radeonfb */
		vxres = xres + xoffset;
		vyres = yres + yoffset;
	}
	else if (vxres == -1) {
		vxres = (info->smem.len * 8) / (vyres * bpp);
	}
	else if (vyres == -1) {
		vyres = (info->smem.len * 8) / (vxres * bpp);
	}

	if (vxres < xres + xoffset)
		vxres = xres + xoffset;

	if (vyres < yres + yoffset)
		vyres = yres + yoffset;

	if (((vxres * vyres * bpp) / 8) > info->smem.len) {
		R128_ERR("not enough memory for mode %ux%ux%u\n",
				vxres, vyres, bpp);
		return -EINVAL;
	}

	layout->bits_per_pixel = bpp;
	layout->display_width = vxres;
	layout->display_height = vyres;
	layout->x_offset = xoffset;
	layout->y_offset = yoffset;
	layout->pixel_code = depth;

	switch (layout->pixel_code) {
	case 4:  format = 1; break;
	case 8:  format = 2; break;
	case 15: format = 3; break;
	case 16: format = 4; break;
	case 24: format = 5; break;
	case 32: format = 6; break;
	default:
		return -EINVAL;
	}

	switch (info->bios_display) {
	case R128_BIOS_DISPLAY_FP:
		hsync_fudge = hsync_fudge_fp[format - 1];
		break;
	case R128_BIOS_DISPLAY_FP_CRT:
		hsync_fudge = hsync_fudge_fp_crt[format - 1];
		break;
	case R128_BIOS_DISPLAY_CRT:
	default:
		hsync_fudge = hsync_fudge_default[format - 1];
		break;
	}

	save->crtc_gen_cntl = R128_CRTC_EXT_DISP_EN | R128_CRTC_EN |
		(format << 8) |
		((var->vmode & FB_VMODE_DOUBLE) ? R128_CRTC_DBL_SCAN_EN : 0) |
		((var->vmode & FB_VMODE_INTERLACED) ? R128_CRTC_INTERLACE_EN : 0) |
		((var->sync & FB_SYNC_COMP_HIGH_ACT) ? R128_CRTC_CSYNC_EN : 0);

	save->crtc_ext_cntl = R128_VGA_ATI_LINEAR | R128_XCRT_CNT_EN;
	save->dac_cntl = R128_DAC_MASK_ALL | R128_DAC_VGA_ADR_EN |
		R128_DAC_8BIT_EN;	/* FIXME: disable DAC_VGA_ADR_EN? */

	if ((info->is_dfp) && (!info->is_pro2)) {
		if (info->panel_xres < xres)
			xres = info->panel_xres;
		if (info->panel_yres < yres)
			yres = info->panel_yres;
		/* FIXME */
		BUG();
	}

	h_total = (xres + var->right_margin + var->hsync_len + var->left_margin);
	v_total = (yres + var->upper_margin + var->vsync_len + var->lower_margin);

	save->crtc_h_total_disp = (((h_total / 8) - 1) & 0xffff) |
		(((xres / 8) - 1) << 16); /* FIXME: wrong order? */

	hsync_wid = var->hsync_len / 8;

	if (!hsync_wid)
		hsync_wid = 0x01;
	else if (hsync_wid > 0x3f)
		hsync_wid = 0x3f;

	hsync_start = ((((xres + var->right_margin) >> 3) - 1) << 3) | hsync_fudge;

	save->crtc_h_sync_strt_wid = (hsync_start & 0xfff) | (hsync_wid << 16) |
		((var->sync & FB_SYNC_HOR_HIGH_ACT) ? 0 : R128_CRTC_H_SYNC_POL);

	save->crtc_v_total_disp = ((v_total - 1) & 0xffff) | ((yres - 1) << 16);

	vsync_wid = var->vsync_len;

	if (!vsync_wid)
		vsync_wid = 1;
	if (vsync_wid > 0x1f)
		vsync_wid = 0x1f;

	vsync_start = yres + var->lower_margin;

	save->crtc_v_sync_strt_wid = ((vsync_start - 1) & 0xfff) |
		(vsync_wid << 16) |
		((var->sync & FB_SYNC_VERT_HIGH_ACT) ? 0 : R128_CRTC_V_SYNC_POL);

	save->crtc_offset = 0;
	
	if ((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_NOW)
		save->crtc_offset_cntl = 0x00010000;
	else
		save->crtc_offset_cntl = 0;

	save->crtc_pitch = layout->display_width / 8;

	R128_DEBUG("Pitch = %d bytes (displayWidth = %d)\n",
			save->crtc_pitch, layout->display_width);

	/* disable endian swap and vga ram */
	save->config_cntl &= ~0x103;

#if defined(__BIG_ENDIAN)
	switch (layout->pixel_code) {
	case 15:
	case 16: save->config_cntl |= APER_0_BIG_ENDIAN_16BPP_SWAP; break;
	case 32: save->config_cntl |= APER_0_BIG_ENDIAN_32BPP_SWAP; break;
	default:
		 break;
	}
#endif

	return 0;
}

static
void r128fb_init_fp_registers(struct r128_info_rec *info, struct r128_save_rec *save, const struct fb_var_screeninfo *var)
{
	struct r128_save_rec *orig = &info->saved_reg;
	u16 xres = var->xres;
	u16 yres = var->yres;
	u32 hratio, vratio;

	if (info->bios_display == R128_BIOS_DISPLAY_CRT) {
		save->crtc_ext_cntl |= R128_CRTC_CRT_ON;
		save->crtc2_gen_cntl = 0;
		save->fp_gen_cntl = orig->fp_gen_cntl;
		save->fp_gen_cntl &= ~(R128_FP_FPON |
				R128_FP_CRTC_USE_SHADOW_VEND |
				R128_FP_CRTC_HORZ_DIV2_EN |
				R128_FP_CRTC_HOR_CRT_DIV2_DIS |
				R128_FP_USE_SHADOW_EN);
		save->fp_gen_cntl |= (R128_FP_SEL_CRTC2 |
				R128_FP_CRTC_DONT_SHADOW_VPAR);
		save->fp_panel_cntl = orig->fp_panel_cntl & ~R128_FP_DIGON;
		save->lvds_gen_cntl = orig->lvds_gen_cntl & ~(R128_LVDS_ON |
				R128_LVDS_BLON);
		return;
	}

	if (xres > info->panel_xres)
		xres = info->panel_xres;
	if (yres > info->panel_yres)
		yres = info->panel_yres;

	hratio = r128fb_div(xres * R128_HORZ_STRETCH_RATIO_MAX,
			info->panel_xres);
	vratio = r128fb_div(yres * R128_VERT_STRETCH_RATIO_MAX,
			info->panel_yres);

	save->fp_horz_stretch = (((hratio & R128_HORZ_STRETCH_RATIO_MASK) <<
				R128_HORZ_STRETCH_RATIO_SHIFT) |
			(orig->fp_horz_stretch & (R128_HORZ_PANEL_SIZE |
						R128_HORZ_FP_LOOP_STRETCH |
						R128_HORZ_STRETCH_RESERVED)));
	save->fp_horz_stretch &= ~R128_HORZ_AUTO_RATIO_FIX_EN;
	save->fp_horz_stretch &= ~R128_AUTO_HORZ_RATIO;

	if (xres == info->panel_xres)
		save->fp_horz_stretch &= ~(R128_HORZ_STRETCH_BLEND |
				R128_HORZ_STRETCH_ENABLE);
	else
		save->fp_horz_stretch |= (R128_HORZ_STRETCH_BLEND |
				R128_HORZ_STRETCH_ENABLE);

	save->fp_vert_stretch = (((vratio & R128_VERT_STRETCH_RATIO_MASK) <<
			R128_VERT_STRETCH_RATIO_SHIFT) |
			(orig->fp_vert_stretch & (R128_VERT_PANEL_SIZE |
					   R128_VERT_STRETCH_RESERVED)));
	save->fp_vert_stretch &= ~R128_VERT_AUTO_RATIO_EN;
	if (yres == info->panel_yres)
		save->fp_vert_stretch &= ~(R128_VERT_STRETCH_ENABLE |
				R128_VERT_STRETCH_BLEND);
	else
		save->fp_vert_stretch |= (R128_VERT_STRETCH_ENABLE |
				R128_VERT_STRETCH_BLEND);

	save->fp_gen_cntl = (orig->fp_gen_cntl & ~(R128_FP_SEL_CRTC2 |
						R128_FP_CRTC_USE_SHADOW_VEND |
						R128_FP_CRTC_HORZ_DIV2_EN |
						R128_FP_CRTC_HOR_CRT_DIV2_DIS |
						R128_FP_USE_SHADOW_EN));

	save->fp_panel_cntl = orig->fp_panel_cntl;
	save->lvds_gen_cntl = orig->lvds_gen_cntl;
	save->tmds_crc = orig->tmds_crc;

	if (!info->is_dfp) {
		if (info->bios_display == R128_BIOS_DISPLAY_FP_CRT) {
			save->crtc_ext_cntl |= R128_CRTC_CRT_ON;
		}
		else {
			save->crtc_ext_cntl &= ~R128_CRTC_CRT_ON;
			save->dac_cntl |= R128_DAC_CRT_SEL_CRTC2;
			save->crtc2_gen_cntl = 0;
		}
	}

	if (info->is_dfp) {
		save->fp_gen_cntl = orig->fp_gen_cntl;
		save->fp_gen_cntl &= ~(R128_FP_CRTC_USE_SHADOW_VEND |
					R128_FP_CRTC_USE_SHADOW_ROWCUR |
					R128_FP_CRTC_HORZ_DIV2_EN |
					R128_FP_CRTC_HOR_CRT_DIV2_DIS |
					R128_FP_CRT_SYNC_SEL |
					R128_FP_USE_SHADOW_EN);
		
		save->fp_panel_cntl  |= (R128_FP_DIGON | R128_FP_BLON);
		save->fp_gen_cntl    |= (R128_FP_FPON | R128_FP_TDMS_EN |
					R128_FP_CRTC_DONT_SHADOW_VPAR |
					R128_FP_CRTC_DONT_SHADOW_HEND);
		save->tmds_transmitter_cntl = (orig->tmds_transmitter_cntl
				 & ~R128_TMDS_PLLRST) | R128_TMDS_PLLEN;
	}
	else {
		save->lvds_gen_cntl |= (R128_LVDS_ON | R128_LVDS_BLON);
	}

	save->fp_crtc_h_total_disp = save->crtc_h_total_disp;
	save->fp_crtc_v_total_disp = save->crtc_v_total_disp;
	save->fp_h_sync_strt_wid = save->crtc_h_sync_strt_wid;
	save->fp_v_sync_strt_wid = save->crtc_v_sync_strt_wid;
}

static
void r128fb_init_pll_registers(struct r128_info_rec *info, struct r128_save_rec *save, u32 dot_clock)
{
	struct r128_pll_rec *pll = &info->pll;
	unsigned long freq;
	struct {
		int divider;
		int bitvalue;
	} *post_div, post_divs[] = {
		{ 1, 0 },
		{ 2, 1 },
		{ 4, 2 },
		{ 8, 3 },
		{ 3, 4 },
		{ 6, 6 },
		{ 12, 7 },
		{ 0, 0 }
	};

	freq = r128fb_div(100000000, dot_clock);

	if (freq > pll->max_pll_freq)
		freq = pll->max_pll_freq;
	if (freq * 12 < pll->min_pll_freq)
		freq = pll->min_pll_freq / 12;

	for (post_div = &post_divs[0]; post_div->divider != 0; ++post_div) {
		save->pll_output_freq = post_div->divider * freq;
		if ((save->pll_output_freq >= pll->min_pll_freq) &&
			(save->pll_output_freq <= pll->max_pll_freq))
			break;
	}

	save->dot_clock_freq = freq;
	save->feedback_div = r128fb_div(pll->reference_div * save->pll_output_freq, pll->reference_freq);
	save->post_div = post_div->divider;

	R128_DEBUG("dc=%d, of=%d, fd=%d, pd=%d\n",
			save->dot_clock_freq,
			save->pll_output_freq,
			save->feedback_div,
			save->post_div);

	save->ppll_ref_div = pll->reference_div;
	save->ppll_div_3 = (save->feedback_div | (post_div->bitvalue << 16));
	save->htotal_cntl = 0;
}


static
int r128fb_init_dda_registers(struct r128_info_rec *info, struct r128_save_rec *save, const struct fb_var_screeninfo *var)
{
	struct r128_pll_rec *pll = &info->pll;
	int display_fifo_width = 128;
	int display_fifo_depth = 32;
	int xclk_freq = pll->xclk;
	int vclk_freq = r128fb_div(pll->reference_freq * save->feedback_div, pll->reference_div * save->post_div);
	int xclks_per_transfer;
	int xclks_per_transfer_precise;
	int useable_precision;
	int roff;
	int ron;

	if ((info->is_dfp) && (!info->is_pro2))
		if (info->panel_xres != var->xres)
			vclk_freq = (vclk_freq * var->xres) / info->panel_xres;

	xclks_per_transfer = r128fb_div(xclk_freq * display_fifo_width, vclk_freq * var->bits_per_pixel);
	useable_precision = __ilog2(xclks_per_transfer) + 2;
	xclks_per_transfer_precise = r128fb_div((xclk_freq * display_fifo_width) << (11 - useable_precision), vclk_freq * var->bits_per_pixel);
	roff = xclks_per_transfer_precise * (display_fifo_depth - 4);
	ron = (4 * info->ram->MB +
		3 * max(info->ram->Trcd - 2, 0) +
		2 * info->ram->Trp +
		info->ram->Twr +
		info->ram->CL +
		info->ram->Tr2w +
		xclks_per_transfer) << (11 - useable_precision);

	if (ron + info->ram->Rloop >= roff) {
		R128_ERR("(Ron = %d) + (Rloop = %d) >= (Roff = %d)\n",
				ron, info->ram->Rloop, roff);
		return -EINVAL;
	}

	save->dda_config = (xclks_per_transfer_precise |
			(useable_precision << 16) |
			(info->ram->Rloop << 20));
	save->dda_on_off = (ron << 16) | roff;

	R128_DEBUG("XclkFreq = %d; VclkFreq = %d; per = %d, %d (useable = %d)\n",
			xclk_freq, vclk_freq, xclks_per_transfer,
			xclks_per_transfer_precise, useable_precision);

	R128_DEBUG("Roff = %d, Ron = %d, Rloop = %d\n",
			roff, ron, info->ram->Rloop);

	return 0;
}

/**
 * internal framebuffer helper routines
 */

static
int r128fb_crtc_to_var(struct r128_save_rec *save, struct fb_var_screeninfo *var)
{
	u32 h_total = save->crtc_h_total_disp & 0x1ff;
	u32 h_disp = (save->crtc_h_total_disp >> 16) & 0xff;
	u32 h_sync_strt = (save->crtc_h_sync_strt_wid >> 3) & 0x1ff;
	u32 h_sync_dly = save->crtc_h_sync_strt_wid & 7;
	u32 h_sync_wid = (save->crtc_h_sync_strt_wid >> 16) & 0x3f;
	u32 v_total = save->crtc_v_total_disp & 0x7ff;
	u32 v_disp = (save->crtc_v_total_disp >> 16) & 0x7ff;
	u32 v_sync_strt = save->crtc_v_sync_strt_wid & 0x7ff;
	u32 v_sync_wid = (save->crtc_v_sync_strt_wid >> 16) & 0x1f;
	u32 pix_width = (save->crtc_gen_cntl >> 8) & 7;

	switch (pix_width) {
	case 1:
		var->bits_per_pixel = 4;
		var->red.offset = 0;
		var->red.length = 4;
		var->green.offset = 0;
		var->green.length = 4;
		var->blue.offset = 0;
		var->blue.length = 4;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	case 2:
		var->bits_per_pixel = 8;
		var->red.offset = 0;
		var->red.length = 8;
		var->green.offset = 0;
		var->green.length = 8;
		var->blue.offset = 0;
		var->blue.length = 8;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	case 3:
		var->bits_per_pixel = 16;
		var->red.offset = 10;
		var->red.length = 5;
		var->green.offset = 5;
		var->green.length = 5;
		var->blue.offset = 0;
		var->blue.length = 5;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	case 4:
		var->bits_per_pixel = 16;
		var->red.offset = 11;
		var->red.length = 5;
		var->green.offset = 5;
		var->green.length = 6;
		var->blue.offset = 0;
		var->blue.length = 5;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	case 5:
		var->bits_per_pixel = 24;
		var->red.offset = 16;
		var->red.length = 8;
		var->green.offset = 8;
		var->green.length = 8;
		var->blue.offset = 0;
		var->blue.length = 8;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	case 6:
		var->bits_per_pixel = 32;
		var->red.offset = 16;
		var->red.length = 8;
		var->green.offset = 8;
		var->green.length = 8;
		var->blue.offset = 0;
		var->blue.length = 8;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	default:
		return -EINVAL;
	}

	var->xres = (h_disp + 1) << 3;
	var->yres = v_disp + 1;
	var->left_margin = ((h_total - h_sync_strt - h_sync_wid) << 3) - h_sync_dly;
	var->right_margin = ((h_sync_strt - h_disp) << 3) + h_sync_dly;
	var->upper_margin = v_total - v_sync_strt - v_sync_wid;
	var->lower_margin = v_sync_strt - v_disp;
	var->hsync_len = h_sync_wid << 3;
	var->vsync_len = v_sync_wid;
	var->sync = ((save->crtc_gen_cntl & R128_CRTC_CSYNC_EN) ? FB_SYNC_COMP_HIGH_ACT : 0) |
		((save->crtc_h_sync_strt_wid & R128_CRTC_H_SYNC_POL) ? 0 : FB_SYNC_HOR_HIGH_ACT) |
		((save->crtc_v_sync_strt_wid & R128_CRTC_V_SYNC_POL) ? 0 : FB_SYNC_VERT_HIGH_ACT);
	var->vmode = ((save->crtc_gen_cntl & R128_CRTC_DBL_SCAN_EN) ? FB_VMODE_DOUBLE : 0) |
		((save->crtc_gen_cntl & R128_CRTC_INTERLACE_EN) ? FB_VMODE_INTERLACED : 0);

	return 0;
}

static
void r128fb_fp_to_var(struct r128_save_rec *save, struct fb_var_screeninfo *var)
{
	/* TODO */
}

static
void r128fb_pll_to_var(struct r128_save_rec *save, struct fb_var_screeninfo *var)
{
	var->pixclock = r128fb_div(100000000, save->dot_clock_freq);
}

static
int r128fb_decode_var(struct r128_info_rec *info, struct fb_var_screeninfo *var)
{
	struct r128_save_rec *save = &info->mode_reg;
	int err;

	r128fb_init_common_registers(info, save);

	if ((err = r128fb_init_crtc_registers(info, save, var)) < 0)
		return err;

	if ((info->has_panel_regs) || (info->is_dfp))
		r128fb_init_fp_registers(info, save, var);

	if (var->pixclock > 0) {
		r128fb_init_pll_registers(info, save, var->pixclock);
		if ((err = r128fb_init_dda_registers(info, save, var)) < 0)
			return err;
	}
	else {
		save->ppll_ref_div = info->saved_reg.ppll_ref_div;
		save->ppll_div_3 = info->saved_reg.ppll_div_3;
		save->htotal_cntl = info->saved_reg.htotal_cntl;
		save->dda_config = info->saved_reg.dda_config;
		save->dda_on_off = info->saved_reg.dda_on_off;
	}

	if (var->accel_flags & FB_ACCELF_TEXT)
		info->accel_flags = FB_ACCELF_TEXT;
	else
		info->accel_flags = 0;

	return 0;
}

static
int r128fb_encode_var(struct r128_info_rec *info, struct fb_var_screeninfo *var)
{
	struct r128_save_rec *save = &info->mode_reg;
	struct r128_layout *layout = &info->layout;
	int err;

	if ((err = r128fb_crtc_to_var(save, var)) < 0)
		return err;

	r128fb_fp_to_var(save, var);
	r128fb_pll_to_var(save, var);

	var->xres_virtual = layout->display_width;
	var->yres_virtual = layout->display_height;

	var->red.msb_right = 0;
	var->green.msb_right = 0;
	var->blue.msb_right = 0;
	var->transp.msb_right = 0;
	var->nonstd = 0;
	var->activate = 0;
	var->height = -1;
	var->width = -1;

	var->accel_flags = info->accel_flags;

	return 0;
}

static
void r128fb_encode_fix(struct r128_info_rec *info, struct fb_fix_screeninfo *fix)
{
	struct r128_layout *layout = &info->layout;

	memset(fix, 0, sizeof(struct fb_fix_screeninfo));
	strcpy(fix->id, r128fb_name);
	fix->smem_start = info->smem.addr;
	fix->smem_len = info->smem.len;
	fix->type = FB_TYPE_PACKED_PIXELS;
	fix->type_aux = 0;

	if (layout->bits_per_pixel <= 8)
		fix->visual = FB_VISUAL_PSEUDOCOLOR;
	else
		fix->visual = FB_VISUAL_DIRECTCOLOR;

	fix->xpanstep = 8;
	fix->ypanstep = 1;
	fix->ywrapstep = 0;
	fix->line_length = (layout->display_width * layout->bits_per_pixel) / 8;
	fix->mmio_start = info->mmio.addr;
	fix->mmio_len = info->mmio.len;
	fix->accel = FB_ACCEL_ATI_RAGE128;
}

static
void r128fb_set_dispsw(struct r128_info_rec *info, struct display *disp)
{
	struct r128_layout *layout = &info->layout;

	switch (layout->bits_per_pixel) {
	case 1:
		disp->dispsw = info->dispsw_1bpp;
		break;
	case 4:
		disp->dispsw = info->dispsw_4bpp;
		break;
	case 8:
		disp->dispsw = info->dispsw_8bpp;
		break;
	case 16:
		disp->dispsw = info->dispsw_16bpp;
		disp->dispsw_data = info->cmap.cfb16;
		break;
	case 24:
		disp->dispsw = info->dispsw_24bpp;
		disp->dispsw_data = info->cmap.cfb24;
		break;
	case 32:
		disp->dispsw = info->dispsw_32bpp;
		disp->dispsw_data = info->cmap.cfb32;
		break;
	default:
		disp->dispsw = &fbcon_dummy;
		break;
	}
}

static
int r128fb_getcolreg(u32 regno, u32 *r, u32 *g, u32 *b, u32 *a, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;

	if (regno > 255)
		return 1;

	*r = (info->palette[regno].r << 8) | info->palette[regno].r;
	*g = (info->palette[regno].g << 8) | info->palette[regno].g;
	*b = (info->palette[regno].b << 8) | info->palette[regno].b;
	*a = 0;

	return 0;
}

static
int r128fb_setcolreg(u32 regno, u32 r, u32 g, u32 b, u32 a, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;
	struct r128_layout *layout = &info->layout;
	u32 palreg;
	u32 i;

	if (regno > 255)
		return 1;

	r >>= 8;
	g >>= 8;
	b >>= 8;

	info->palette[regno].r = r;
	info->palette[regno].g = g;
	info->palette[regno].b = b;

	/* initialize gamma-ramp for hi-color+ */
	if ((layout->bits_per_pixel > 8) && (regno == 0))
		for (i = 0; i < 256; i++)
			r128fb_outpal(info, i, i, i, i);

	if (layout->bits_per_pixel == 16)
		palreg = regno * 8;
	else
		palreg = regno;

	if (layout->pixel_code == 16) {
		r128fb_outpal(info, palreg / 2, info->palette[regno / 2].r,
				g, info->palette[regno / 2].b);
		g = info->palette[regno * 2].g;
	}

	if ((layout->pixel_code <= 8) || (regno < 32))
		r128fb_outpal(info, palreg, r, g, b);

	if (regno >= 16)
		return 0;

	switch (layout->pixel_code) {
#if defined(FBCON_HAS_CFB16)
	case 15:
		info->cmap.cfb16[regno] = (regno << 10) | (regno << 5) | regno;
		break;
	case 16:
		info->cmap.cfb16[regno] = (regno << 11) | (regno << 5) | regno;
		break;
#endif
#if defined(FBCON_HAS_CFB24)
	case 24:
		info->cmap.cfb24[regno] = (regno << 16) | (regno << 8) | regno;
		break;
#endif
#if defined(FBCON_HAS_CFB32)
	case 32:
		i = (regno << 8) | regno;
		info->cmap.cfb32[regno] = (i << 16) | i;
		break;
#endif
	default:
		break;
	}

	return 0;
}

static
struct fb_cmap *r128fb_select_cmap(struct r128_info_rec *info)
{
	struct r128_layout *layout = &info->layout;
	struct display *disp;
	u16 num_colors;

	if (info->con < 0)
		disp = &info->disp;
	else
		disp = &fb_display[info->con];

	if (disp->cmap.len) /* non default color map */
		return &disp->cmap;

	if (layout->pixel_code <= 8)
		num_colors = 256;
	else
		num_colors = 32;

	return fb_default_cmap(num_colors);
}

static
void r128fb_install_cmap(struct r128_info_rec *info)
{
	struct fb_cmap *cmap = r128fb_select_cmap(info);

	if ((info->has_panel_regs) || (info->is_dfp))
		r128fb_pal_select(info, 0);

	fb_set_cmap(cmap, 1, r128fb_setcolreg, &info->fb_info);
}

static
void r128fb_get_vblank(struct r128_info_rec *info, struct fb_vblank *vblank)
{
	memset(vblank, 0, sizeof(struct fb_vblank));
	vblank->flags = FB_VBLANK_HAVE_VSYNC;
}

/**
 * framebuffer file operations
 */

static
int r128fb_get_fix(struct fb_fix_screeninfo *fix, int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;

	R128_DEBUG("%s(%p, %d, %p)\n", __FUNCTION__, fix, con, fb);

	r128fb_encode_fix(info, fix);

	return 0;
}

static
int r128fb_get_var(struct fb_var_screeninfo *var, int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;
	
	R128_DEBUG("%s(%p, %d, %p)\n", __FUNCTION__, var, con, fb);

	*var = (con < 0) ? info->disp.var : fb_display[con].var;

	return 0;
}

static
int r128fb_set_var(struct fb_var_screeninfo *var, int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;
	struct display *disp;
	int oldxres, oldyres, oldvxres, oldvyres, oldbpp, oldgreen, oldaccel;
	int var_changed;
	int err;

	R128_DEBUG("%s(%p, %d, %p)\n", __FUNCTION__, var, con, fb);

	disp = (con < 0) ? info->fb_info.disp : &fb_display[con];

	if (!var->xres)
		var->xres = 1;
	if (!var->yres)
		var->yres = 1;
	if (var->xres > var->xres_virtual)
		var->xres_virtual = var->xres;
	if (var->yres > var->yres_virtual)
		var->yres_virtual = var->yres;

#if defined(FBCON_HAS_MFB)
	if (var->bits_per_pixel <= 1)
		var->bits_per_pixel = 1;
	else
#endif
#if defined(FBCON_HAS_CFB4)
	if (var->bits_per_pixel <= 4)
		var->bits_per_pixel = 4;
	else
#endif
#if defined(FBCON_HAS_CFB8)
	if (var->bits_per_pixel <= 8)
		var->bits_per_pixel = 8;
	else
#endif
#if defined(FBCON_HAS_CFB16)
	if (var->bits_per_pixel <= 16)
		var->bits_per_pixel = 16;
	else
#endif
#if defined(FBCON_HAS_CFB24)
	if (var->bits_per_pixel <= 24)
		var->bits_per_pixel = 24;
	else
#endif
#if defined(FBCON_HAS_CFB32)
	if (var->bits_per_pixel <= 32)
		var->bits_per_pixel = 32;
	else
#endif
		return -EINVAL;

	if ((err = r128fb_decode_var(info, var)) < 0)
		return err;

	r128fb_encode_var(info, var);

	if ((var->activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_TEST)
		return 0;

	oldxres = disp->var.xres;
	oldyres = disp->var.yres;
	oldvxres = disp->var.xres_virtual;
	oldvyres = disp->var.yres_virtual;
	oldbpp = disp->var.bits_per_pixel;
	oldgreen = disp->var.green.length;
	oldaccel = disp->var.accel_flags;
	disp->var = *var;

	var_changed = (oldxres != var->xres ||
		oldyres != var->yres ||
		oldvxres != var->xres_virtual ||
		oldvyres != var->yres_virtual ||
		oldgreen != var->green.length ||
		oldbpp != var->bits_per_pixel ||
		oldaccel != var->accel_flags);

	if (var_changed) {
		struct fb_fix_screeninfo fix;
		r128fb_encode_fix(info, &fix);

		disp->screen_base = info->smem.ptr;
		disp->visual = fix.visual;
		disp->type = fix.type;
		disp->type_aux = fix.type_aux;
		disp->ypanstep = fix.ypanstep;
		disp->ywrapstep = fix.ywrapstep;
		disp->line_length = fix.line_length;
		disp->can_soft_blank = 1;
		disp->inverse = 0;

		r128fb_set_dispsw(info, disp);

		if (var->accel_flags & FB_ACCELF_TEXT)
			disp->scrollmode = SCROLL_YNOMOVE;
		else
			disp->scrollmode = SCROLL_YREDRAW;

		if (info->fb_info.changevar)
			info->fb_info.changevar(con);
	}

	if ((!fb->display_fg) || (fb->display_fg->vc_num == con)) {
		r128fb_restore(info, &info->mode_reg);
		if (var->accel_flags & FB_ACCELF_TEXT) {
			if (info->engine_init)
				info->engine_init(info);
		}
	}

	if ((oldbpp != var->bits_per_pixel) ||
		(oldgreen != var->green.length)) {
			if ((err = fb_alloc_cmap(&disp->cmap, 0, 0)))
				return err;
			r128fb_install_cmap(info);
	}

	return 0;
}

static
int r128fb_get_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;

	R128_DEBUG("%s(%p, %d, %d, %p)\n", __FUNCTION__, cmap, kspc, con, fb);

	if (con == info->con) /* current console */
		return fb_get_cmap(cmap, kspc, r128fb_getcolreg, fb);

	fb_copy_cmap(r128fb_select_cmap(info), cmap, kspc ? 0 : 2);

	return 0;
}

static
int r128fb_set_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;
	struct display *disp = (con < 0) ? &info->disp : &fb_display[con];
	u16 cmap_len = (disp->var.bits_per_pixel == 8) ? 256 : 32;
	int err;

	R128_DEBUG("%s(%p, %d, %d, %p)\n", __FUNCTION__, cmap, kspc, con, fb);

	if (disp->cmap.len != cmap_len) {
		err = fb_alloc_cmap(&disp->cmap, cmap_len, 0);

		if (!disp->cmap.len) {
			cmap_len = (disp->var.bits_per_pixel <= 8) ? 256 : 32;
			err = fb_alloc_cmap(&disp->cmap, cmap_len, 0);
		}

		if (err)
			return err;
	}

	/* current console? */
	if (con == info->con) {
		if ((info->has_panel_regs) || (info->is_dfp))
			r128fb_pal_select(info, 0);
		return fb_set_cmap(cmap, kspc, r128fb_setcolreg, fb);
	}
	else {
		fb_copy_cmap(cmap, &disp->cmap, kspc ? 0 : 1);
		return 0;
	}
}

static
int r128fb_pan_display(struct fb_var_screeninfo *var, int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;

	R128_DEBUG("%s(%p, %d, %p)\n", __FUNCTION__, var, con, fb);

	if ((var->xoffset + var->xres > var->xres_virtual) ||
		(var->yoffset + var->yres > var->yres_virtual))
		return -EINVAL;

	OUTREG(R128_CRTC_OFFSET, ((var->yoffset * var->xres_virtual +
		var->xoffset) * var->bits_per_pixel / 8) & ~7);

	return 0;
}

static
int r128fb_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg, int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;

	R128_DEBUG("%s(%p, %p, %u, %lu, %d, %p)\n",
		__FUNCTION__, inode, file, cmd, arg, con, fb);

	switch (cmd) {
	case FBIOGET_VBLANK:
	{
		struct fb_vblank vblank;

		r128fb_get_vblank(info, &vblank);

		if (copy_to_user((void *)arg, &vblank, sizeof(struct fb_vblank)))
			return -EFAULT;

		break;
	}

	case FBIO_ALLOC:
	case FBIO_FREE:
	case FBIOGET_GLYPH:
	case FBIOGET_HWCINFO:
	case FBIOPUT_MODEINFO:
	case FBIOGET_DISPINFO:
		return -EOPNOTSUPP;

	case FBIO_WAITFORVSYNC:
	{
		int i;

		if (get_user(i, (int *)arg))
			return -EFAULT;

		if (i != 0)
			return -EINVAL;

		return r128fb_wait_for_vertical_sync(info);
	}

	default:
		return -EOPNOTSUPP;
	}

	return 0;
}

static
int r128fb_rasterimg(struct fb_info *fb, int start)
{
	R128_DEBUG("%s(%p, %d)\n", __FUNCTION__, fb, start);
	return 0;
}

static
struct fb_ops r128fb_ops = {
	.owner = THIS_MODULE,
	.fb_open = NULL,
	.fb_release = NULL,
	.fb_get_fix = r128fb_get_fix,
	.fb_get_var = r128fb_get_var,
	.fb_set_var = r128fb_set_var,
	.fb_get_cmap = r128fb_get_cmap,
	.fb_set_cmap = r128fb_set_cmap,
	.fb_pan_display = r128fb_pan_display,
	.fb_ioctl = r128fb_ioctl,
	.fb_mmap = NULL,
	.fb_rasterimg = r128fb_rasterimg,
};

static
int r128fbcon_switch(int con, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;

	R128_DEBUG("%s(%d, %p)\n", __FUNCTION__, con, fb);

	/* Do we have to save the colormap? */
	if (fb_display[info->con].cmap.len)
		fb_get_cmap(&fb_display[info->con].cmap, 1, r128fb_getcolreg, fb);

	/* set the current console and switch mode */
	info->con = con;
	return r128fb_set_var(&fb_display[con].var, con, fb);
}

static
int r128fbcon_updatevar(int con, struct fb_info *fb)
{
	R128_DEBUG("%s(%d, %p)\n", __FUNCTION__, con, fb);

	if (con < 0)
		return -EINVAL;

	return r128fb_pan_display(&fb_display[con].var, con, fb);
}

static
void r128fbcon_blank(int blank, struct fb_info *fb)
{
	struct r128_info_rec *info = (struct r128_info_rec *) fb;

	R128_DEBUG("%s(%d, %p)\n",
		__FUNCTION__, blank, fb);

	/* substract one to let vesa modes match. strange thing */
	r128fb_blank(info, blank - 1);
}



/**
 * pci functions
 */

static struct pci_device_id r128fb_id_table[] __devinitdata = {
	R128_PCI_ID(LE, HAS_PANEL_REGS | SDR128_1_1),
	R128_PCI_ID(LF, HAS_PANEL_REGS | SDR128_1_1),
	R128_PCI_ID(MF, HAS_PANEL_REGS | SDR128_1_1),
	R128_PCI_ID(ML, HAS_PANEL_REGS | SDR128_1_1),
	R128_PCI_ID(PA, IS_DFP),
	R128_PCI_ID(PB, IS_DFP),
	R128_PCI_ID(PC, IS_DFP),
	R128_PCI_ID(PD, IS_DFP),
	R128_PCI_ID(PE, IS_DFP),
	R128_PCI_ID(PF, IS_DFP),
	R128_PCI_ID(PG, IS_DFP),
	R128_PCI_ID(PH, IS_DFP),
	R128_PCI_ID(PI, IS_DFP),
	R128_PCI_ID(PJ, IS_DFP),
	R128_PCI_ID(PK, IS_DFP),
	R128_PCI_ID(PL, IS_DFP),
	R128_PCI_ID(PM, IS_DFP),
	R128_PCI_ID(PN, IS_DFP),
	R128_PCI_ID(PO, IS_DFP),
	R128_PCI_ID(PP, IS_DFP),
	R128_PCI_ID(PQ, IS_DFP),
	R128_PCI_ID(PR, IS_DFP),
	R128_PCI_ID(PS, IS_DFP),
	R128_PCI_ID(PT, IS_DFP),
	R128_PCI_ID(PU, IS_DFP),
	R128_PCI_ID(PV, IS_DFP),
	R128_PCI_ID(PW, IS_DFP),
	R128_PCI_ID(PX, IS_DFP),
	R128_PCI_ID(RE, SDR128_1_1),
	R128_PCI_ID(RF, SDR128_1_1),
	R128_PCI_ID(RG, SDR128_1_1),
	R128_PCI_ID(RI, 0),	/* does this one exist? */
	R128_PCI_ID(RK, 0),
	R128_PCI_ID(RL, 0),
	R128_PCI_ID(SE, 0),
	R128_PCI_ID(SF, 0),
	R128_PCI_ID(SG, 0),
	R128_PCI_ID(SH, 0),
	R128_PCI_ID(SK, 0),
	R128_PCI_ID(SL, 0),
	R128_PCI_ID(SM, 0),
	R128_PCI_ID(SN, 0),
	R128_PCI_ID(TF, IS_PRO2 | SDR128_1_1),
	R128_PCI_ID(TL, IS_PRO2 | SDR128_1_1),
	R128_PCI_ID(TR, IS_PRO2 | SDR128_1_1),
	R128_PCI_ID(TS, IS_PRO2),
	R128_PCI_ID(TT, IS_PRO2),
	R128_PCI_ID(TU, IS_PRO2),
	{ 0, }
};

MODULE_DEVICE_TABLE(pci, r128fb_id_table);

static
int __devinit r128fb_probe(struct pci_dev *dev, const struct pci_device_id *id)
{
	struct r128_info_rec *info;
	int err;

	if ((err = pci_enable_device(dev)))
		return err;

	if ((err = pci_request_regions(dev, r128fb_name)))
		return err;

	info = kmalloc(sizeof(struct r128_info_rec), GFP_KERNEL);

	if (!info) {
		R128_ERR("kmalloc failed\n");
		err = -ENOMEM;
		goto probe_failed_release;
	}

	memset(info, 0, sizeof(struct r128_info_rec));

	info->mmio.addr = pci_resource_start(dev, 2) & 0xffffc000;
	info->mmio.len = pci_resource_len(dev, 2);
	info->mmio.ptr = ioremap_nocache(info->mmio.addr, info->mmio.len);

	if (!info->mmio.ptr) {
		R128_ERR("ioremap mmio failed\n");
		err = -EIO;
		goto probe_failed_free;
	}

	info->smem.addr = pci_resource_start(dev, 0) & 0xffffff00;
	info->smem.len = INREG(R128_CONFIG_MEMSIZE);
	info->smem.ptr = ioremap(info->smem.addr, info->smem.len);

	if (!info->smem.ptr) {
		R128_ERR("ioremap smem failed\n");
		err = -EIO;
		goto probe_failed_iounmap_mmio;
	}

	info->mem_cntl = INREG(R128_MEM_CNTL);
	info->bus_cntl = INREG(R128_BUS_CNTL);

	info->dev = dev;
	info->has_panel_regs = !!(id->driver_data & HAS_PANEL_REGS);
	info->is_dfp = !!(id->driver_data & IS_DFP);
	info->is_pro2 = !!(id->driver_data & IS_PRO2);

	if ((err = r128fb_get_bios_parameters(info)) < 0) {
		R128_ERR("video rom not found\n");
		goto probe_failed_iounmap_smem;
	}

	if ((err = r128fb_get_pll_parameters(info)) < 0) {
		R128_ERR("pll parameters not found\n");
		goto probe_failed_iounmap_bios;
	}

	if (info->has_panel_regs)
		info->bios_display = R128_BIOS_DISPLAY_FP;
	else
		info->bios_display = R128_BIOS_DISPLAY_CRT;

	switch (info->mem_cntl & 3) {
	case 0:
		if (id->driver_data & SDR128_1_1)
			info->ram = &r128fb_ram_type[0];
		else
			info->ram = &r128fb_ram_type[1];
		break;
	case 1:
		info->ram = &r128fb_ram_type[2];
		break;
	case 2:
		info->ram = &r128fb_ram_type[3];
		break;
	default:
		info->ram = &r128fb_ram_type[1];
		break;
	}

	pci_set_drvdata(dev, info);

	strcpy(info->fb_info.modename, r128fb_name);
	info->fb_info.node = NODEV;
	info->fb_info.flags = FBINFO_FLAG_DEFAULT;
	info->fb_info.fbops = &r128fb_ops;
	info->fb_info.disp = &info->disp;
	info->fb_info.changevar = NULL;
	info->fb_info.switch_con = r128fbcon_switch;
	info->fb_info.updatevar = r128fbcon_updatevar;
	info->fb_info.blank = r128fbcon_blank;
	info->con = -1;

#if defined(FBCON_HAS_MFB)
	info->dispsw_1bpp = &fbcon_mfb;
#endif
#if defined(FBCON_HAS_CFB4)
	info->dispsw_4bpp = &fbcon_cfb4;
#endif
#if defined(FBCON_HAS_CFB8)
	info->dispsw_8bpp = &fbcon_cfb8;
#endif
#if defined(FBCON_HAS_CFB16)
	info->dispsw_16bpp = &fbcon_cfb16;
#endif
#if defined(FBCON_HAS_CFB24)
	info->dispsw_24bpp = &fbcon_cfb24;
#endif
#if defined(FBCON_HAS_CFB32)
	info->dispsw_32bpp = &fbcon_cfb32;
#endif

	/* save current mode, so it can be restored on removal */
	r128fb_save(info, &info->saved_reg);

	/*
	 * ensure that there are no uninitialized values in mode_reg,
	 * but do not use the saved palette
	 */
	memcpy(&info->mode_reg, &info->saved_reg, sizeof(struct r128_save_rec));
	info->mode_reg.palette_valid = 0;

	/* clear framebuffer */
	memset(info->smem.ptr, 0, info->smem.len);

	/* gui idle and vertical sync irq */
	init_waitqueue_head(&info->gui_idle_wait);
	init_waitqueue_head(&info->vsync_wait);

	/* check whether interrupts are available */
	if (INREG8(R128_INTERRUPT_PIN) & 0x01) {
		if (request_irq(dev->irq, r128fb_irq, SA_SHIRQ, r128fb_name, info)) {
			R128_ERR("request_irq failed\n");
		}
		else {
			info->gen_int_cntl = /* R128_GUI_IDLE_INT | */
				R128_VSYNC_INT;
		}
	}

	/* set default mode */
	if ((err = r128fb_set_var(&r128fb_default, -1, &info->fb_info)) < 0) {
		R128_ERR("could not set default mode\n");
		goto probe_failed_erase_devdata;
	}

	if ((err = register_framebuffer(&info->fb_info)) < 0) {
		R128_ERR("register_framebuffer failed\n");
		goto probe_failed_restore_mode;
	}

	R128_INFO("fb%d: %s frame buffer device\n",
		GET_FB_IDX(info->fb_info.node), r128fb_name);

#if defined(CONFIG_MTRR)
	info->mtrr_reg = mtrr_add(info->smem.addr, info->smem.len, MTRR_TYPE_WRCOMB, 1);
	R128_INFO("MTRR turned %s\n", (info->mtrr_reg < 0) ? "off" : "on");
#endif

#if !defined(R128FB_ACCEL_MODULE)
	if (!r128fb_accel_init_single(GET_FB_IDX(info->fb_info.node)))
		R128_ERR("failed to enable console acceleration\n");
#endif
	return 0;

probe_failed_restore_mode:
	r128fb_restore(info, &info->saved_reg);
probe_failed_erase_devdata:
	pci_set_drvdata(dev, NULL);
probe_failed_iounmap_bios:
	iounmap(info->bios.ptr);
probe_failed_iounmap_smem:
	iounmap(info->smem.ptr);
probe_failed_iounmap_mmio:
	iounmap(info->mmio.ptr);
probe_failed_free:
	kfree(info);
probe_failed_release:
	pci_release_regions(dev);

	return err;
}

static
void __devexit r128fb_remove(struct pci_dev *dev)
{
	struct r128_info_rec *info = pci_get_drvdata(dev);

	if (info) {
		if (info->gen_int_cntl)
			free_irq(dev->irq, info);

		r128fb_restore(info, &info->saved_reg);

		if (unregister_framebuffer(&info->fb_info) < 0)
			R128_CRIT("unregister_framebuffer failed\n");
		else
			give_up_console(&fb_con);
#if defined(CONFIG_MTRR)
		if (info->mtrr_reg >= 0)
			mtrr_del(info->mtrr_reg, info->smem.addr, info->smem.len);
#endif
		if (info->bios.ptr)
			iounmap(info->bios.ptr);
		if (info->smem.ptr)
			iounmap(info->smem.ptr);
		if (info->mmio.ptr)
			iounmap(info->mmio.ptr);

		kfree(info);
		pci_set_drvdata(dev, NULL);
	}

	pci_release_regions(dev);
}

static
int r128fb_save_state(struct pci_dev *dev, u32 state)
{
	R128_INFO("%s(%p, %u)\n", __FUNCTION__, dev, state);
	return 0;
}

static
int r128fb_suspend(struct pci_dev *dev, u32 state)
{
	R128_INFO("%s(%p, %u)\n", __FUNCTION__, dev, state);
	return 0;
}

static
int r128fb_resume(struct pci_dev *dev)
{
	R128_INFO("%s(%p)\n", __FUNCTION__, dev);
	return 0;
}

static
int r128fb_enable_wake(struct pci_dev *dev, u32 state, int enable)
{
	R128_INFO("%s(%p, %u, %d)\n", __FUNCTION__, dev, state, enable);
	return 0;
}

static
struct pci_driver r128fb_pci_driver = {
	.name = "ATI Rage128",
	.id_table = r128fb_id_table,
	.probe = r128fb_probe,
	.remove = __devexit_p(r128fb_remove),
	.save_state = r128fb_save_state,
	.suspend = r128fb_suspend,
	.resume = r128fb_resume,
	.enable_wake = r128fb_enable_wake,
};

int __init r128fb_module_init(void)
{
	return pci_module_init(&r128fb_pci_driver);
}

void __exit r128fb_module_exit(void)
{
	pci_unregister_driver(&r128fb_pci_driver);
}

module_init(r128fb_module_init);
module_exit(r128fb_module_exit);

MODULE_AUTHOR("Andreas Oberritter <obi@saftware.de>");
MODULE_DESCRIPTION("ATI Rage128 Framebuffer Driver");
MODULE_LICENSE("GPL");
