/*
 * $Id: r128fb_base.h,v 1.6 2003/08/19 06:25:14 obi Exp $
 *
 * ATI Rage 128 Framebuffer Driver
 *
 * Copyright (C) 2003 Andreas Oberritter <obi@saftware.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * This driver has been widely inspired by the Rage 128 drivers of
 * XFree86 4.x, Linux 2.4.x and svgalib 1.9.x. Thanks to everybody
 * who contributed to them.
 *
 */

#ifndef _R128FB_BASE_H
#define _R128FB_BASE_H

#include <linux/fb.h>
#include <linux/pci.h>
#include <video/fbcon.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb24.h>
#include <video/fbcon-cfb32.h>
#include <video/fbcon-mfb.h>

/* enable to print informational driver messages */
#define R128_VERBOSE_MESSAGES	1

/* enable to print debugging driver messages */
#define R128_DEBUG_MESSAGES	0

/* enable to print register reads and writes */
#define R128_REGISTER_DEBUG	0

/* enable to print every bios memory read */
#define R128_BIOS_DEBUG		0

struct r128_ram_rec {		/* All values in XCLKS    */
	int ML;			/* Memory Read Latency    */
	int MB;			/* Memory Burst Length    */
	int Trcd;		/* RAS to CAS delay       */
	int Trp;		/* RAS percentage         */
	int Twr;		/* Write Recovery         */
	int CL;			/* CAS Latency            */
	int Tr2w;		/* Read to Write Delay    */
	int Rloop;		/* Loop Latency           */
	int Rloop_fudge;	/* Add to ML to get Rloop */
};

struct r128_save_rec {
	/* Common registers */
	u32 ovr_clr;
	u32 ovr_wid_left_right;
	u32 ovr_wid_top_bottom;
	u32 ov0_scale_cntl;
	u32 mpp_tb_config;
	u32 mpp_gp_config;
	u32 subpic_cntl;
	u32 viph_control;
	u32 i2c_cntl_1;
	u32 gen_int_cntl;
	u32 cap0_trig_cntl;
	u32 cap1_trig_cntl;
	u32 bus_cntl;
	u32 config_cntl;

	/* Other registers to save for VT switches */
	u32 dp_datatype;
	u32 gen_reset_cntl;
	u32 clock_cntl_index;
	u32 amcgpio_en_reg;
	u32 amcgpio_mask;

	/* CRTC registers */
	u32 crtc_gen_cntl;
	u32 crtc_ext_cntl;
	u32 dac_cntl;
	u32 crtc_h_total_disp;
	u32 crtc_h_sync_strt_wid;
	u32 crtc_v_total_disp;
	u32 crtc_v_sync_strt_wid;
	u32 crtc_offset;
	u32 crtc_offset_cntl;
	u32 crtc_pitch;

	/* CRTC2 registers */
	u32 crtc2_gen_cntl;

	/* Flat panel registers */
	u32 fp_crtc_h_total_disp;
	u32 fp_crtc_v_total_disp;
	u32 fp_gen_cntl;
	u32 fp_h_sync_strt_wid;
	u32 fp_horz_stretch;
	u32 fp_panel_cntl;
	u32 fp_v_sync_strt_wid;
	u32 fp_vert_stretch;
	u32 lvds_gen_cntl;
	u32 tmds_crc;
	u32 tmds_transmitter_cntl;

	/* Computed values for PLL */
	u32 dot_clock_freq;
	u32 pll_output_freq;
	int feedback_div;
	int post_div;

	/* PLL registers */
	u32 ppll_ref_div;
	u32 ppll_div_3;
	u32 htotal_cntl;

	/* DDA register */
	u32 dda_config;
	u32 dda_on_off;

	/* Palette */
	u8 palette_valid;
	u32 palette[256];
};

struct r128_pll_rec {
	u16 reference_freq;
	u16 reference_div;
	u32 min_pll_freq;
	u32 max_pll_freq;
	u16 xclk;
};

struct r128_mem_info {
	unsigned long addr;		/* physical address		*/
	unsigned long len;		/* region size			*/
	unsigned char *ptr;		/* data pointer			*/
};

struct r128_layout {
	int bits_per_pixel;
	int depth;
	int display_width;
	int display_height;
	int x_offset;
	int y_offset;
	int pixel_code;
};

struct r128_cursor {
	/* enabled */
	u8 on;
	/* visible size */
	u8 width;
	u8 height;
	/* on-screen location of cursor */
	u16 x;
	u16 y;
	/* off-screen position of cursor bitmap */
	u32 offset;
};

struct r128_edid {
	struct i2c_adapter *adapter;
};

struct r128_iic {
	struct i2c_adapter *adapter;
	u32 r128_i2c_timing;
	u32 r128_M;
	u32 r128_N;
};

struct r128_info_rec {
	struct fb_info fb_info;
	struct display disp;
	struct pci_dev *dev;
	u32 video_ram;

	struct r128_mem_info smem;	/* Frame buffer			*/
	struct r128_mem_info mmio;	/* MMIO region			*/
	struct r128_mem_info bios;	/* Video BIOS			*/

	u32 mem_cntl;
	u32 bus_cntl;

	u8 bios_display;		/* Display device set by BIOS	*/
	u8 has_panel_regs;		/* Chip can connect to a FP	*/
	int fp_bios_start;		/* Start of flat panel info	*/

	/* Computed values for FPs */
	int panel_xres;
	int panel_yres;
	int h_over_plus;
	int h_sync_width;
	int h_blank;
	int v_over_plus;
	int v_sync_width;
	int v_blank;
	int panel_pwr_dly;

	/* current console number */
	int con;

	/* current framebuffer layout */
	struct r128_layout layout;

	/* cursor settings */
	struct r128_cursor cursor;

	/* edid data */
	struct r128_edid edid;

	/* video iic data */
	u32 i2c_cntl_1;
	struct r128_iic iic;

	/* irq stuff */
	u32 gen_int_cntl;
	int irq_enabled;
	int vsync_irq_count;
	wait_queue_head_t vsync_wait;
	wait_queue_head_t gui_idle_wait;

	struct {
		u8 r;
		u8 g;
		u8 b;
	} palette[256];

	/* color maps */
	union {
		u16 cfb16[16];
		u32 cfb24[16];
		u32 cfb32[16];
	} cmap;

	struct r128_pll_rec pll;
	struct r128_ram_rec *ram;
	struct r128_save_rec saved_reg;	/* Original (text) mode		*/
	struct r128_save_rec mode_reg;	/* Current mode			*/

	/* Free slots in the FIFO (64 max) */
	int fifo_slots;

	/* Computed values for Rage 128 */
	int pitch;
	int datatype;
	u32 dp_gui_master_cntl;

	u8 is_dfp;
	u8 is_pro2;

	int mtrr_reg;

	/* acceleration hooks */
	u32 accel_flags;
	struct display_switch *dispsw_1bpp;
	struct display_switch *dispsw_4bpp;
	struct display_switch *dispsw_8bpp;
	struct display_switch *dispsw_16bpp;
	struct display_switch *dispsw_24bpp;
	struct display_switch *dispsw_32bpp;
	void (*engine_init)(struct r128_info_rec *info);
};

#define _R128_BIOS8(v)				\
	(info->bios.ptr[v])
#define _R128_BIOS16(v)				\
	(info->bios.ptr[v] |			\
	(info->bios.ptr[(v) + 1] << 8))
#define _R128_BIOS32(v)				\
	(info->bios.ptr[v] |			\
	(info->bios.ptr[(v) + 1] << 8) |	\
	(info->bios.ptr[(v) + 2] << 16) |	\
	(info->bios.ptr[(v) + 3] << 24))

#if R128_BIOS_DEBUG
#define R128_BIOS8(v) ({					\
		unsigned char ___x = _R128_BIOS8(v);		\
		printk(KERN_DEBUG "B(%04x,%02x)\n", v, ___x);	\
		___x;						\
		})
#define R128_BIOS16(v) ({					\
		unsigned short ___x = _R128_BIOS16(v);		\
		printk(KERN_DEBUG "B(%04x,%04x)\n", v, ___x);	\
		___x;						\
		})
#define R128_BIOS32(v) ({					\
		unsigned int ___x = _R128_BIOS32(v);		\
		printk(KERN_DEBUG "B(%04x,%08x)\n", v, ___x);	\
		___x;						\
		})
#else	/* R128_BIOS_DEBUG */
#define R128_BIOS8(v)	_R128_BIOS8(v)
#define R128_BIOS16(v)	_R128_BIOS16(v)
#define R128_BIOS32(v)	_R128_BIOS32(v)
#endif	/* R128_BIOS_DEBUG */

/* bits for pci private data */
#define SDR128_1_1	0x00000001UL
#define SDR64_1_1	0x00000002UL
#define SDR64_2_1	0x00000004UL
#define DDR64		0x00000008UL
#define HAS_PANEL_REGS	0x00000010UL
#define IS_DFP		0x00000020UL
#define IS_PRO2		0x00000040UL

/*
 * avoid having all this redundant information
 * repeating in the source code
 */
#define R128_PCI_ID(id, flags) {	\
	PCI_VENDOR_ID_ATI,		\
	PCI_DEVICE_ID_ATI_RAGE128_##id,	\
	PCI_ANY_ID,			\
	PCI_ANY_ID,			\
	PCI_CLASS_DISPLAY_VGA << 8,	\
	0xffff00,			\
	flags				\
}

/* some additional pci ids */
#define PCI_DEVICE_ID_ATI_RAGE128_MF	0x4d46
#define PCI_DEVICE_ID_ATI_RAGE128_ML	0x4d4c
#define PCI_DEVICE_ID_ATI_RAGE128_PA	0x5041
#define PCI_DEVICE_ID_ATI_RAGE128_PB	0x5042
#define PCI_DEVICE_ID_ATI_RAGE128_PC	0x5043
#define PCI_DEVICE_ID_ATI_RAGE128_PD	0x5044
#define PCI_DEVICE_ID_ATI_RAGE128_PE	0x5045
#define PCI_DEVICE_ID_ATI_RAGE128_SE	0x5345
#define PCI_DEVICE_ID_ATI_RAGE128_SF	0x5346
#define PCI_DEVICE_ID_ATI_RAGE128_SG	0x5347
#define PCI_DEVICE_ID_ATI_RAGE128_SH	0x5348
#define PCI_DEVICE_ID_ATI_RAGE128_SK	0x534b
#define PCI_DEVICE_ID_ATI_RAGE128_SL	0x534c
#define PCI_DEVICE_ID_ATI_RAGE128_SM	0x534d
#define PCI_DEVICE_ID_ATI_RAGE128_SN	0x534e
#define PCI_DEVICE_ID_ATI_RAGE128_TF	0x5446
#define PCI_DEVICE_ID_ATI_RAGE128_TL	0x544c
#define PCI_DEVICE_ID_ATI_RAGE128_TS	0x5453
#define PCI_DEVICE_ID_ATI_RAGE128_TT	0x5454
#define PCI_DEVICE_ID_ATI_RAGE128_TU	0x5455

/* Fall out of wait loops after this count */
#define R128_TIMEOUT  2000000

#define R128_PRINTK(type, fmt, args...)	\
	printk(type "r128fb: " fmt, ##args)

#define R128_CRIT(fmt, args...)		R128_PRINTK(KERN_CRIT, fmt, ##args)
#define R128_ERR(fmt, args...)		R128_PRINTK(KERN_ERR, fmt, ##args)
#define R128_WARNING(fmt, args...)	R128_PRINTK(KERN_WARNING, fmt, ##args)
#define R128_NOTICE(fmt, args...)	R128_PRINTK(KERN_NOTICE, fmt, ##args)

#if R128_VERBOSE_MESSAGES
#define R128_INFO(fmt, args...)		R128_PRINTK(KERN_INFO, fmt, ##args)
#else
#define R128_INFO(fmt, args...)
#endif

#if R128_DEBUG_MESSAGES
#define R128_DEBUG(fmt, args...)	R128_PRINTK(KERN_DEBUG, fmt, ##args)
#else
#define R128_DEBUG(fmt, args...)
#endif

#ifndef FBIO_WAITFORVSYNC
#define FBIO_WAITFORVSYNC	_IOW('F', 0x20, int)
#endif

#endif /* _R128FB_BASE_H */
