/*
 * $Id$
 *
 * ATI Rage 128 Framebuffer Driver
 *
 * Copyright (C) 2003 Andreas Oberritter <obi@saftware.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * This driver has been widely inspired by the Rage 128 drivers of
 * XFree86 4.x, Linux 2.4.x and svgalib 1.9.x. Thanks to everybody
 * who contributed to them.
 *
 */

#include <linux/i2c.h>
#include <linux/i2c-algo-bit.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>

#include "r128fb_inline.h"
#include "r128fb_reg.h"

#define DDC_REG	R128_GPIO_MONID
#define SET_SDA	R128_GPIO_MONID_EN_0
#define SET_SCL	R128_GPIO_MONID_EN_3
#define GET_SDA	R128_GPIO_MONID_Y_0
#define GET_SCL	R128_GPIO_MONID_Y_3
#define MSK_SDA R128_GPIO_MONID_MASK_0
#define MSK_SCL R128_GPIO_MONID_MASK_3
#define A_SDA	R128_GPIO_MONID_A_0
#define A_SCL	R128_GPIO_MONID_A_3

static void r128fb_edid_setsda(void *data, int state)
{
	struct r128_info_rec *info = data;

	if (state)
		OUTREGP(DDC_REG, 0, ~SET_SDA);
	else
		OUTREGP(DDC_REG, SET_SDA, ~SET_SDA);
}

static void r128fb_edid_setscl(void *data, int state)
{
	struct r128_info_rec *info = data;

	if (state)
		OUTREGP(DDC_REG, 0, ~SET_SCL);
	else
		OUTREGP(DDC_REG, SET_SCL, ~SET_SCL);
}

static int r128fb_edid_getsda(void *data)
{
	struct r128_info_rec *info = data;

	return !!(INREG(DDC_REG) & GET_SDA);
}

static int r128fb_edid_getscl(void *data)
{
	struct r128_info_rec *info = data;

	return !!(INREG(DDC_REG) & GET_SCL);
}

static struct i2c_algo_bit_data r128fb_edid_data = {
	.data = NULL,
	.setsda = r128fb_edid_setsda,
	.setscl = r128fb_edid_setscl,
	.getsda = r128fb_edid_getsda,
	.getscl = r128fb_edid_getscl,
	.udelay = 100,
	.mdelay = 100,
	.timeout = 1000,
};

static void r128fb_edid_inc_use(struct i2c_adapter *adapter)
{
	MOD_INC_USE_COUNT;
}

static void r128fb_edid_dec_use(struct i2c_adapter *adapter)
{
	MOD_DEC_USE_COUNT;
}

static int r128fb_edid_client_register(struct i2c_client *client)
{
	R128_INFO("r128fb_iic_client_register(%p)\n", client);
	return 0;
}

static int r128fb_edid_client_unregister(struct i2c_client *client)
{
	R128_INFO("r128fb_iic_client_unregister(%p)\n", client);
	return 0;
}

static struct i2c_adapter r128fb_edid_adapter = {
	.name = "ATI Rage128 EDID IIC Adapter",
	.id = I2C_ALGO_ATI,
	.algo = NULL,
	.algo_data = NULL,
	.inc_use = r128fb_edid_inc_use,
	.dec_use = r128fb_edid_dec_use,
	.client_register = r128fb_edid_client_register,
	.client_unregister = r128fb_edid_client_unregister,
	.data = NULL,
};

int r128fb_edid_init_single(int index)
{
	struct r128_info_rec *info;
	struct r128_edid *edid;

	if (strncmp(registered_fb[index]->modename, "ATI Rage128", 11))
		return 0;

	info = (struct r128_info_rec *) registered_fb[index];
	edid = &info->edid;

	if (edid->adapter != NULL) {
		R128_ERR("edid->adapter != NULL\n");
		return 0;
	}

	edid->adapter = kmalloc(sizeof(struct i2c_adapter), GFP_KERNEL);
	memcpy(edid->adapter, &r128fb_edid_adapter, sizeof(struct i2c_adapter));

	edid->adapter->algo_data = kmalloc(sizeof(struct i2c_algo_bit_data), GFP_KERNEL);
	memcpy(edid->adapter->algo_data, &r128fb_edid_data, sizeof(struct i2c_algo_bit_data));

	((struct i2c_algo_bit_data *)edid->adapter->algo_data)->data = info;

	OUTREGP(DDC_REG, MSK_SDA | MSK_SCL, 0xffffffff);
	OUTREGP(DDC_REG, 0, ~(A_SDA | A_SCL));

	if (i2c_bit_add_bus(edid->adapter)) {
		kfree(edid->adapter->algo_data);
		kfree(edid->adapter);
		edid->adapter = NULL;
		return 0;
	}

	return 1;
}

void r128fb_edid_exit_single(int index)
{
	struct r128_info_rec *info;
	struct r128_edid *edid;

	if (strncmp(registered_fb[index]->modename, "ATI Rage128", 11))
		return;

	info = (struct r128_info_rec *) registered_fb[index];
	edid = &info->edid;

	if (edid->adapter) {
		i2c_bit_del_bus(edid->adapter);
		kfree(edid->adapter->algo_data);
		kfree(edid->adapter);
		edid->adapter = NULL;
	}
}

int r128fb_edid_init(void)
{
	int count = 0;
	int i;

	for (i = 0; i < num_registered_fb; i++)
		count += r128fb_edid_init_single(i);

	if (!count)
		return -ENODEV;

	return 0;
}

void r128fb_edid_exit(void)
{
	int i;

	for (i = 0; i < num_registered_fb; i++)
		r128fb_edid_exit_single(i);
}

module_init(r128fb_edid_init);
module_exit(r128fb_edid_exit);

MODULE_AUTHOR("Andreas Oberritter <obi@saftware.de>");
MODULE_DESCRIPTION("ATI Rage128 Framebuffer EDID IIC Driver");
MODULE_LICENSE("GPL");
