/*
 * $Id: r128fb_iic.c,v 1.1 2003/08/19 06:25:14 obi Exp $
 *
 * ATI Rage 128 Framebuffer Driver
 *
 * Copyright (C) 2003 Andreas Oberritter <obi@saftware.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * This driver has been widely inspired by the Rage 128 drivers of
 * XFree86 4.x, Linux 2.4.x and svgalib 1.9.x. Thanks to everybody
 * who contributed to them.
 *
 */

#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>

#include "r128fb_accel.h"
#include "r128fb_inline.h"
#include "r128fb_reg.h"

#define I2C_DONE	0x00000001
#define I2C_NACK	0x00000002
#define I2C_HALT	0x00000004
#define I2C_SOFT_RST	0x00000020
#define I2C_DRIVE_EN	0x00000040
#define I2C_DRIVE_SEL	0x00000080
#define I2C_START	0x00000100
#define I2C_STOP	0x00000200
#define I2C_RECEIVE	0x00000400
#define I2C_ABORT	0x00000800
#define I2C_GO		0x00001000

#define I2C_SEL		0x00010000
#define I2C_EN		0x00020000

static u8 r128fb_iic_wait_for_ack(struct r128_info_rec *info)
{
	int i;
	u8 i2c_cntl_0;

	mdelay(1);

	for (i = 0; i < 1000000; i++) {
		i2c_cntl_0 = INREG8(R128_I2C_CNTL_0);
		if (i2c_cntl_0 & I2C_HALT)
			return I2C_HALT;
		if (i2c_cntl_0 & I2C_NACK)
			return I2C_NACK;
		if (i2c_cntl_0 & I2C_DONE)
			return I2C_DONE;
	}

	R128_ERR("r128fb_iic_wait_for_ack: timeout\n");
	return I2C_HALT;
}

static void r128fb_iic_halt(struct r128_info_rec *info)
{
	int i;

	/* reset status flags */
	OUTREG8(R128_I2C_CNTL_0, INREG8(R128_I2C_CNTL_0) & 0xf8);

	/* issue abort call */
	OUTREG8(R128_I2C_CNTL_0 + 1, INREG8(R128_I2C_CNTL_0 + 1) | 0x18);

	/* wait for go bit to go low */
	for (i = 0; i < 1000000; i++)
		if (!(INREG8(R128_I2C_CNTL_0 + 1) & (I2C_GO >> 8)))
			break;
}

static int r128fb_iic_master_xfer(struct i2c_adapter *adapter, struct i2c_msg msgs[], int num)
{
	struct r128_info_rec *info = adapter->algo_data;
	struct r128_iic *iic = &info->iic;
	u32 i2c_cntl_0, i2c_cntl_1;
	u16 addr;
	int i, j;

	for (i = 0; i < num; i++) {
		if (msgs[i].len > 16)
			return -EINVAL;

		i2c_cntl_0 = (iic->r128_N << 24) | (iic->r128_M << 16) | I2C_GO | I2C_START | I2C_DRIVE_EN;
		i2c_cntl_1 = (iic->r128_i2c_timing << 24) | I2C_EN | I2C_SEL | (1 << 8) | msgs[i].len;

		if (msgs[i].flags & I2C_M_RD) {
			addr = msgs[i].addr | 1;
			i2c_cntl_0 |= I2C_RECEIVE;
		}
		else {
			addr = msgs[i].addr & ~1;
		}

		if ((i + 1) == num)
			i2c_cntl_0 |= I2C_STOP;

		r128fb_wait_for_fifo(info, msgs[i].len + 4);

		/* clear the status bits of the I2C controller */
		OUTREG(R128_I2C_CNTL_0, I2C_DONE | I2C_NACK | I2C_HALT | I2C_SOFT_RST);

		/* write the address into the buffer first */
		OUTREG(R128_I2C_DATA, addr);

		/* write value into the buffer */
		if (!(addr & 1))
			for (j = 0; j < msgs[i].len; j++)
				OUTREG8(R128_I2C_DATA, msgs[i].buf[j]);

		/* save register to prevent erasing it on a mode change */
		info->i2c_cntl_1 = i2c_cntl_1;

		OUTREG(R128_I2C_CNTL_1, i2c_cntl_1);
		OUTREG(R128_I2C_CNTL_0, i2c_cntl_0);

		while (INREG8(R128_I2C_CNTL_0 + 1) & (I2C_GO >> 8));

		if (r128fb_iic_wait_for_ack(info) != I2C_DONE) {
			r128fb_iic_halt(info);
			break;
		}

		/* read value from the buffer */
		if (addr & 1)
			for (j = 0; j < msgs[i].len; j++)
				msgs[i].buf[j] = INREG8(R128_I2C_DATA);
	}

	return i;
}

static int r128fb_iic_algo_control(struct i2c_adapter *adapter, unsigned int cmd, unsigned long arg)
{
	return 0;
}

static u32 r128fb_iic_functionality(struct i2c_adapter *adapter)
{
	return I2C_FUNC_I2C;
}

static struct i2c_algorithm r128fb_iic_algorithm = {
	.name = "ATI Rage128 Video IIC Algorithm",
	.id = I2C_ALGO_ATI,
	.master_xfer = r128fb_iic_master_xfer,
	.smbus_xfer = NULL,
	.slave_send = NULL,
	.slave_recv = NULL,
	.algo_control = r128fb_iic_algo_control,
	.functionality = r128fb_iic_functionality,
};

static void r128fb_iic_inc_use(struct i2c_adapter *adapter)
{
	MOD_INC_USE_COUNT;
}

static void r128fb_iic_dec_use(struct i2c_adapter *adapter)
{
	MOD_DEC_USE_COUNT;
}

static int r128fb_iic_client_register(struct i2c_client *client)
{
	R128_INFO("r128fb_iic_client_register(%p)\n", client);
	return 0;
}

static int r128fb_iic_client_unregister(struct i2c_client *client)
{
	R128_INFO("r128fb_iic_client_unregister(%p)\n", client);
	return 0;
}

static struct i2c_adapter r128fb_iic_adapter = {
	.name = "ATI Rage128 Video IIC Adapter",
	.id = I2C_ALGO_ATI,
	.algo = &r128fb_iic_algorithm,
	.algo_data = NULL,
	.inc_use = r128fb_iic_inc_use,
	.dec_use = r128fb_iic_dec_use,
	.client_register = r128fb_iic_client_register,
	.client_unregister = r128fb_iic_client_unregister,
	.data = NULL,
};

int r128fb_iic_init_single(int index)
{
	struct r128_info_rec *info;
	struct r128_iic *iic;
	struct r128_pll_rec *pll;
	int nm;

	if (strncmp(registered_fb[index]->modename, "ATI Rage128", 11))
		return 0;

	info = (struct r128_info_rec *) registered_fb[index];
	iic = &info->iic;
	pll = &info->pll;

	if (iic->adapter != NULL) {
		R128_ERR("iic->adapter != NULL\n");
		return 0;
	}

	iic->adapter = kmalloc(sizeof(struct i2c_adapter), GFP_KERNEL);
	memcpy(iic->adapter, &r128fb_iic_adapter, sizeof(struct i2c_adapter));

#define I2C_CLOCK_FREQ 80000
	nm = r128fb_div(pll->reference_freq * 10000, 4 * I2C_CLOCK_FREQ);

	for (iic->r128_N = 1; iic->r128_N < 255; iic->r128_N++)
		if ((iic->r128_N * (iic->r128_N - 1)) > nm)
			break;

	iic->r128_M = iic->r128_N - 1;
	iic->r128_i2c_timing = 2 * iic->r128_N;
	iic->adapter->algo_data = info;
	iic->adapter->timeout = 100;
	iic->adapter->retries = 3;

	/* reset iic */
	OUTREG8(R128_I2C_CNTL_1 + 2, (I2C_SEL | I2C_EN) >> 16);
	OUTREG8(R128_I2C_CNTL_0, I2C_DONE | I2C_NACK | I2C_HALT | I2C_SOFT_RST | I2C_DRIVE_EN | I2C_DRIVE_SEL);

	if (i2c_add_adapter(iic->adapter)) {
		kfree(iic->adapter);
		iic->adapter = NULL;
		return 0;
	}

	return 1;
}

void r128fb_iic_exit_single(int index)
{
	struct r128_info_rec *info;
	struct r128_iic *iic;

	if (strncmp(registered_fb[index]->modename, "ATI Rage128", 11))
		return;

	info = (struct r128_info_rec *) registered_fb[index];
	iic = &info->iic;

	if (iic->adapter) {
		i2c_del_adapter(iic->adapter);
		kfree(iic->adapter);
		iic->adapter = NULL;
	}
}

int r128fb_iic_init(void)
{
	int count = 0;
	int i;

	for (i = 0; i < num_registered_fb; i++)
		count += r128fb_iic_init_single(i);

	if (!count)
		return -ENODEV;

	return 0;
}

void r128fb_iic_exit(void)
{
	int i;

	for (i = 0; i < num_registered_fb; i++)
		r128fb_iic_exit_single(i);
}

module_init(r128fb_iic_init);
module_exit(r128fb_iic_exit);

MODULE_AUTHOR("Andreas Oberritter <obi@saftware.de>");
MODULE_DESCRIPTION("ATI Rage128 Framebuffer Video IIC Driver");
MODULE_LICENSE("GPL");
