/*
 * $Id: r128fb_inline.h,v 1.5 2003/08/19 07:11:05 obi Exp $
 *
 * ATI Rage 128 Framebuffer Driver
 *
 * Copyright (C) 2003 Andreas Oberritter <obi@saftware.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * This driver has been widely inspired by the Rage 128 drivers of
 * XFree86 4.x, Linux 2.4.x and svgalib 1.9.x. Thanks to everybody
 * who contributed to them.
 *
 */

#ifndef _R128FB_INLINE_H
#define _R128FB_INLINE_H

#include "r128fb_base.h"
#include "r128fb_reg.h"

static inline
int r128fb_div(int n, int d)
{
	return (n + (d / 2)) / d;
}

#if !defined(__powerpc__)
static inline
int __ilog2(unsigned long x)
{
	int i;

	if (x == 0)
		return -1;

	for (i = 0; x != 0; i++)
		x >>= 1;

	return i - 1;
}
#endif

static inline
u8 r128fb_readb(struct r128_info_rec *info, u16 addr)
{
	u8 val = readb(&info->mmio.ptr[addr]);
#if R128_REGISTER_DEBUG
	printk(KERN_DEBUG "R(%04x,%02x)\n", addr, val);
#endif
	return val;
}

static inline
u16 r128fb_readw(struct r128_info_rec *info, u16 addr)
{
	u16 val = readw(&info->mmio.ptr[addr]);
#if R128_REGISTER_DEBUG
	printk(KERN_DEBUG "R(%04x,%04x)\n", addr, val);
#endif
	return val;
}

static inline
u32 r128fb_readl(struct r128_info_rec *info, u16 addr)
{
	u32 val = readl(&info->mmio.ptr[addr]);
#if R128_REGISTER_DEBUG
	printk(KERN_DEBUG "R(%04x,%08x)\n", addr, val);
#endif
	return val;
}

static inline
void r128fb_writeb(struct r128_info_rec *info, u16 addr, u8 val)
{
#if R128_REGISTER_DEBUG
	printk(KERN_DEBUG "W(%04x,%02x)\n", addr, val);
#endif
	writeb(val, &info->mmio.ptr[addr]);
}

static inline
void r128fb_writew(struct r128_info_rec *info, u16 addr, u16 val)
{
#if R128_REGISTER_DEBUG
	printk(KERN_DEBUG "W(%04x,%04x)\n", addr, val);
#endif
	writew(val, &info->mmio.ptr[addr]);
}

static inline
void r128fb_writel(struct r128_info_rec *info, u16 addr, u32 val)
{
#if R128_REGISTER_DEBUG
	printk(KERN_DEBUG "W(%04x,%08x)\n", addr, val);
#endif
	writel(val, &info->mmio.ptr[addr]);
}

#define INREG8(addr)			r128fb_readb(info, addr)
#define INREG16(addr)			r128fb_readw(info, addr)
#define INREG(addr)			r128fb_readl(info, addr)
#define OUTREG8(addr, val)		r128fb_writeb(info, addr, val)
#define OUTREG16(addr, val)		r128fb_writew(info, addr, val)
#define OUTREG(addr, val)		r128fb_writel(info, addr, val)
#define OUTREGP(addr, val, mask)	\
	OUTREG(addr, (INREG(addr) & (mask)) | (val))

static inline
u32 r128fb_inpll(struct r128_info_rec *info, u8 addr)
{
	OUTREG8(R128_CLOCK_CNTL_INDEX, addr & 0x1f);
	return INREG(R128_CLOCK_CNTL_DATA);
}

static inline
void r128fb_outpll(struct r128_info_rec *info, u8 addr, u32 val)
{
	OUTREG8(R128_CLOCK_CNTL_INDEX, ((addr) & 0x1f) | R128_PLL_WR_EN);
	OUTREG(R128_CLOCK_CNTL_DATA, val);
}

static inline
void r128fb_outpllp(struct r128_info_rec *info, u8 addr, u32 val, u32 mask)
{
	u32 tmp = r128fb_inpll(info, addr);
	tmp &= mask;
	tmp |= val;
	r128fb_outpll(info, addr, tmp);
}

static inline
void r128fb_outpal_start(struct r128_info_rec *info, u16 idx)
{
	OUTREG8(R128_PALETTE_INDEX, (idx));
}

static inline
void r128fb_outpal_next(struct r128_info_rec *info, u8 r, u8 g, u8 b)
{
	OUTREG(R128_PALETTE_DATA, ((r) << 16) | ((g) << 8) | (b));
}

static inline
void r128fb_outpal_next32(struct r128_info_rec *info, u32 v)
{
	OUTREG(R128_PALETTE_DATA, (v & 0x00ffffff));
}

static inline
void r128fb_outpal(struct r128_info_rec *info, u16 idx, u8 r, u8 g, u8 b)
{
	r128fb_outpal_start(info, idx);
	r128fb_outpal_next(info, r, g, b);
}

static inline
void r128fb_inpal_start(struct r128_info_rec *info, u16 idx)
{
	OUTREG(R128_PALETTE_INDEX, (idx) << 16);
}

static inline
u32 r128fb_inpal_next(struct r128_info_rec *info)
{
	return INREG(R128_PALETTE_DATA);
}

static inline
void r128fb_pal_select(struct r128_info_rec *info, u8 idx)
{
	u32 tmp = INREG(R128_DAC_CNTL);

	if (idx)
		OUTREG(R128_DAC_CNTL, tmp | R128_DAC_PALETTE_ACC_CTL);
	else
		OUTREG(R128_DAC_CNTL, tmp & ~R128_DAC_PALETTE_ACC_CTL);
}

static inline
void r128fb_pll_wait_read_update_complete(struct r128_info_rec *info)
{
	unsigned long timeout = jiffies + (HZ / 100);

	while (time_before(jiffies, timeout)) {
		if (!(r128fb_inpll(info, R128_PPLL_REF_DIV) & R128_PPLL_ATOMIC_UPDATE_R))
			return;
	}

	R128_ERR("pll write timeout\n");
}

static inline
void r128fb_pll_write_update(struct r128_info_rec *info)
{
	r128fb_outpllp(info, R128_PPLL_REF_DIV, R128_PPLL_ATOMIC_UPDATE_W, 0xffff);
}

#endif /* _R128FB_INLINE_H */
